# Mergin Maps Mobile - Docker Build Guide

## Vue d'ensemble

Ce projet utilise un Dockerfile multi-stage pour construire l'application Mergin Maps Mobile en une seule commande. Le Dockerfile combine l'installation des dépendances de base et la construction de l'application finale.

## Structure du Dockerfile

Le Dockerfile est organisé en deux stages principaux :

### Stage 1: `base`
- Installation d'Ubuntu 24.04
- Installation des dépendances système (CMake, Ninja, OpenJDK 17, etc.)
- Qt 6.6.3 fourni via une archive ZIP (Artifactory) passée par `ZIP_URL`
- Configuration de l'environnement de développement

### Stage 2: `final`
- Installation de l'Android SDK et NDK
- Téléchargement des SDKs Mergin Maps
- Configuration de l'environnement Android
- Construction de l'application avec CMake et Ninja
- Génération de l'APK final

## Stratégies de Tagging

Le système de tagging suit les standards internationaux et offre plusieurs options :

### 1. Tag `latest` (Recommandé pour le développement)
```bash
docker build --add-host=mirror.dev.futurmap.net:192.168.0.109 -t mergin-maps:latest .
```

### 2. Tag basé sur la version (Recommandé pour les releases)
```bash
docker build --add-host=mirror.dev.futurmap.net:192.168.0.109 -t mergin-maps:1.0.0 .
```

### 3. Tag basé sur la date (Format: YYYYMMDD-HHMM)
```bash
docker build --add-host=mirror.dev.futurmap.net:192.168.0.109 -t mergin-maps:$(date +%Y%m%d-%H%M) .
```

### 4. Tag basé sur le commit Git
```bash
docker build --add-host=mirror.dev.futurmap.net:192.168.0.109 -t mergin-maps:commit-$(git rev-parse --short HEAD) .
```

### 5. Tag combiné (Version + Date)
```bash
docker build --add-host=mirror.dev.futurmap.net:192.168.0.109 -t mergin-maps:1.0.0-$(date +%Y%m%d) .
```

## Utilisation du Script de Build

Un script de build automatisé est fourni pour simplifier le processus :

### Installation
```bash
chmod +x build.sh
```

### Options disponibles

#### Build avec tag `latest`
```bash
./build.sh latest
```

#### Build avec tag de version
```bash
VERSION=1.0.0 ./build.sh version
```

#### Build avec tag basé sur la date
```bash
./build.sh date
```

#### Build avec tag basé sur Git
```bash
./build.sh git
```

#### Build avec tag combiné
```bash
VERSION=1.0.0 ./build.sh combined
```

#### Aide
```bash
./build.sh help
```

## Extraction de l'APK

Une fois l'image construite, vous pouvez extraire l'APK final :

### Méthode 1: Copie directe
```bash
# Créer un dossier de sortie
mkdir -p output

# Extraire l'APK
docker run --rm -v $(pwd)/output:/output mergin-maps:latest \
  cp /var/www/ratp-qt/build/app/android-build/build/outputs/apk/release/android-build-release.apk /output/
```

### Méthode 2: Accès interactif
```bash
# Lancer un conteneur interactif
docker run -it --rm mergin-maps:latest bash

# Dans le conteneur, l'APK est disponible à :
# /var/www/ratp-qt/build/app/android-build/build/outputs/apk/release/android-build-release.apk
```

## Configuration du Réseau

Le build nécessite l'accès au miroir interne :
```bash
--add-host=mirror.dev.futurmap.net:192.168.0.109
```

## Optimisations du Build

### Cache Docker
Le Dockerfile est optimisé pour maximiser l'utilisation du cache Docker :
- Les dépendances système sont installées en premier
- Les téléchargements sont séparés des installations
- Les étapes de build sont organisées pour minimiser les invalidations de cache

### Multi-stage Build
- Le stage `base` peut être réutilisé pour d'autres projets
- Le stage `final` ne contient que les éléments nécessaires à la construction
- Réduction de la taille finale de l'image

## Dépannage

### Problèmes de réseau
Si le build échoue à cause du miroir :
```bash
# Vérifier la connectivité
ping mirror.dev.futurmap.net

# Utiliser un miroir alternatif si nécessaire
```

### Problèmes de mémoire
Pour les builds sur des machines avec peu de RAM :
```bash
# Limiter l'utilisation de la mémoire
docker build --memory=4g --add-host=mirror.dev.futurmap.net:192.168.0.109 -t mergin-maps:latest .
```

### Problèmes de stockage
Pour libérer de l'espace après le build :
```bash
# Nettoyer les images non utilisées
docker image prune -f

# Nettoyer tous les éléments non utilisés
docker system prune -f
```

## Métadonnées de l'Image

L'image finale contient les métadonnées suivantes :
- `maintainer`: Mergin Maps Team
- `description`: Mergin Maps Mobile Application - Multi-stage build
- `version`: 1.0.0
- `build-date`: Date de construction automatique

## Migration depuis l'Ancien Système

### Ancien processus (2 étapes)
```bash
# Étape 1: Build de l'image de base
docker build -f Dockerfile.base -t ubuntu_base .

# Étape 2: Build de l'application
docker build --add-host=mirror.dev.futurmap.net:192.168.0.109 -t ratp:1.0 .
```

### Nouveau processus (1 étape)
```bash
# Build complet en une seule commande
docker build --add-host=mirror.dev.futurmap.net:192.168.0.109 -t mergin-maps:1.0.0 .
```

## Support

Pour toute question ou problème :
1. Vérifiez les logs de build Docker
2. Consultez la documentation Mergin Maps
3. Contactez l'équipe de développement

---

**Note**: Ce Dockerfile remplace complètement les anciens `Dockerfile.base` et `Dockerfile`. Les anciens fichiers peuvent être supprimés après validation du nouveau système.

## Qt dependencies via ZIP (Artifactory)

"Documentation (with examples) in English"

This project no longer installs Qt via aqtinstall in the base image. Instead, it downloads a pre-packaged ZIP containing Qt directories and unpacks them into `/root/Qt/${QT_VERSION}`.

### Build arguments

```bash
# Example with Podman
podman build -f Dockerfile.base \
  --build-arg ZIP_URL="https://artifactory.example.com/path/to/qt-dependencies.zip" \
  --build-arg QT_VERSION=6.6.3 \
  -t mergin-maps-base:zip .

# Example with Docker
docker build -f Dockerfile.base \
  --build-arg ZIP_URL="https://artifactory.example.com/path/to/qt-dependencies.zip" \
  --build-arg QT_VERSION=6.6.3 \
  -t mergin-maps-base:zip .
```

### Expected ZIP layout

```text
deps/
  qt/
    6.6.3/                      # must match ${QT_VERSION}
      gcc_64/
        bin/
          qmake
        lib/cmake/Qt6/
      android_arm64_v8a/
        lib/cmake/Qt6/
          qt.toolchain.cmake
```

Minimum required files inside the archive:
- `deps/qt/${QT_VERSION}/gcc_64/bin/qmake`
- `deps/qt/${QT_VERSION}/gcc_64/lib/cmake/Qt6/*`
- `deps/qt/${QT_VERSION}/android_arm64_v8a/lib/cmake/Qt6/qt.toolchain.cmake`

### CI variables (example)

Define `QT_ZIP_URL` in GitLab CI variables and map it to the build arg via `ZIP_URL: "$QT_ZIP_URL"`. Optionally set `QT_VERSION` (defaults to `6.6.3`).

```yaml
variables:
  ZIP_URL: "$QT_ZIP_URL"
  QT_VERSION: "6.6.3"
```

### Troubleshooting

- If the build fails with "Qt directory not found inside archive", verify the ZIP contains `deps/qt/${QT_VERSION}`.
- Ensure the `qt.toolchain.cmake` file exists under the Android ABI directory.
