# Mergin Maps Mobile - GitLab CI/CD Pipeline

## Vue d'ensemble

Ce pipeline CI/CD automatise la construction de l'application Mergin Maps Mobile en utilisant le nouveau Dockerfile multi-stage. Il génère des images Docker avec des tags standardisés et exporte l'APK finale comme artefact téléchargeable.

## Architecture du Pipeline

### Stages du Pipeline

1. **`cleanup`** - Nettoyage des ressources Docker
2. **`docker_build`** - Construction de l'image Docker multi-stage
3. **`apk_export`** - Extraction et sauvegarde de l'APK

### Déclenchement Automatique

Le pipeline se déclenche automatiquement sur :
- **Push** vers la branche `enhance-improve-dockerfile-ci`
- **Merge Request** vers la branche `enhance-improve-dockerfile-ci`
- **Schedule** (pipelines programmés)
- **Web** (déclenchement manuel via l'interface GitLab)
- **API** (déclenchement via API)

**Note** : Le pipeline ne se déclenche **PAS** sur la branche `main` pour éviter les builds automatiques en production.

## Stratégie de Tagging

### Tags Automatiques Générés

Le pipeline génère automatiquement plusieurs tags selon la branche :

#### 1. Tag Principal
- **`enhance-improve-dockerfile-ci`** → `enhance-YYYYMMDD-HHMM`
- **Version** (v1.0.0, 2.1.3) → `1.0.0`, `2.1.3`
- **Release** (release/*) → `release-YYYYMMDD-HHMM`
- **Autres branches** → `commit-SHORTSHA`

#### 2. Tags Supplémentaires
- `commit-$CI_COMMIT_SHORT_SHA` - Tag basé sur le commit
- `branch-$CI_COMMIT_REF_NAME` - Tag basé sur la branche

### Exemples de Tags

```bash
# Branche enhance-improve-dockerfile-ci
mergin-maps:enhance-20241220-1430
mergin-maps:commit-a1b2c3d
mergin-maps:branch-enhance-improve-dockerfile-ci

# Branche de version
mergin-maps:1.0.0
mergin-maps:commit-a1b2c3d
mergin-maps:branch-v1.0.0

# Branche de release
mergin-maps:release-20241220-1430
mergin-maps:commit-a1b2c3d
mergin-maps:branch-release-v1.0.0

# Branche de feature
mergin-maps:commit-a1b2c3d
mergin-maps:branch-feature-new-ui
```

## Configuration des Variables

### Variables Globales

```yaml
variables:
  # Docker configuration
  DOCKER_DRIVER: overlay2
  DOCKER_TLS_CERTDIR: "/certs"
  DOCKER_HOST: tcp://docker:2376
  DOCKER_TLS_VERIFY: 1
  DOCKER_CERT_PATH: "$DOCKER_TLS_CERTDIR/client"
  
  # Image configuration
  IMAGE_NAME: "mergin-maps"
  REGISTRY_URL: "$CI_REGISTRY"
  FULL_IMAGE_NAME: "$CI_REGISTRY/$CI_PROJECT_PATH"
  
  # Build configuration
  MIRROR_HOST: "mirror.dev.futurmap.net:192.168.0.109"
  
  # APK configuration
  APK_SOURCE_PATH: "/var/www/ratp-qt/build/app/android-build/build/outputs/apk/release/android-build-release.apk"
  APK_ARTIFACT_NAME: "mergin-maps-mobile.apk"
  ZIP_URL: "$QT_ZIP_URL"
  QT_VERSION: "6.6.3"
```

### Variables GitLab CI Intégrées

- `$CI_COMMIT_REF_NAME` - Nom de la branche
- `$CI_COMMIT_SHA` - Hash complet du commit
- `$CI_COMMIT_SHORT_SHA` - Hash court du commit
- `$CI_PIPELINE_ID` - ID du pipeline
- `$CI_PROJECT_NAME` - Nom du projet
- `$CI_REGISTRY` - URL du registry GitLab

## Détail des Jobs

### 1. `cleanup_docker`

**Objectif** : Nettoyer les ressources Docker pour optimiser l'espace disque

**Configuration** :
- **Stage** : `cleanup`
- **Image** : `docker:24.0.5`
- **Services** : `docker:24.0.5-dind`
- **Tags** : `build-ratp-apk`
- **Règles** : Déclenché sur push vers `enhance-improve-dockerfile-ci`
- **Allow failure** : `true` (ne bloque pas le pipeline)

**Actions** :
```bash
docker system prune -f
docker image prune -f
```

### 2. `build_docker_image`

**Objectif** : Construire l'image Docker multi-stage avec tags standardisés

**Configuration** :
- **Stage** : `docker_build`
- **Image** : `docker:24.0.5`
- **Services** : `docker:24.0.5-dind`
- **Tags** : `build-ratp-apk`
- **Cache** : Docker cache optimisé

**Actions** :
1. Génération du tag principal selon la branche
2. Construction de l'image avec le nouveau Dockerfile multi-stage
3. Application de multiples tags
4. Sauvegarde des informations d'image pour les stages suivants

**Commande de build** :
```bash
docker build \
  --add-host="mirror.dev.futurmap.net:192.168.0.109" \
  --tag "mergin-maps:$IMAGE_TAG" \
  --tag "mergin-maps:commit-$CI_COMMIT_SHORT_SHA" \
  --tag "mergin-maps:branch-$CI_COMMIT_REF_NAME" \
  --build-arg BUILD_DATE="$(date -u +'%Y-%m-%dT%H:%M:%SZ')" \
  --build-arg VCS_REF="$CI_COMMIT_SHA" \
  --build-arg VERSION="$IMAGE_TAG" \
  --build-arg ZIP_URL="$ZIP_URL" \
  --build-arg QT_VERSION="$QT_VERSION" \
  .
```

### 3. `extract_apk`

**Objectif** : Extraire l'APK finale et la sauvegarder comme artefact

**Configuration** :
- **Stage** : `apk_export`
- **Image** : `docker:24.0.5`
- **Services** : `docker:24.0.5-dind`
- **Tags** : `build-ratp-apk`
- **Dependencies** : `build_docker_image`

**Actions** :
1. Extraction de l'APK depuis le conteneur Docker
2. Vérification de l'existence et de la validité de l'APK
3. Génération de métadonnées JSON
4. Sauvegarde comme artefact téléchargeable

**Artefacts générés** :
- `mergin-maps-mobile.apk` - L'APK finale
- `apk_metadata.json` - Métadonnées du build

## Gestion des Artefacts

### APK Artefact

**Nom** : `mergin-maps-apk-$CI_COMMIT_SHORT_SHA`
**Durée de vie** : 30 jours
**Contenu** :
- `mergin-maps-mobile.apk` - APK finale
- `apk_metadata.json` - Métadonnées du build

### Métadonnées APK

Le fichier `apk_metadata.json` contient :
```json
{
  "project": "mergin-maps-mobile",
  "branch": "main",
  "commit": "a1b2c3d4e5f6...",
  "short_commit": "a1b2c3d",
  "pipeline_id": "12345",
  "build_date": "2024-12-20T14:30:00Z",
  "image_tag": "latest",
  "apk_name": "mergin-maps-mobile.apk",
  "apk_size": "52428800"
}
```

## Optimisations

### Cache Docker

Le pipeline utilise un cache Docker optimisé :
```yaml
cache:
  key: "docker-cache-$CI_COMMIT_REF_SLUG"
  paths:
    - .docker-cache/
  policy: pull-push
```

### Build Args

Arguments de build passés au Dockerfile :
- `BUILD_DATE` - Date de construction ISO 8601
- `VCS_REF` - Hash du commit Git
- `VERSION` - Tag de version

## Configuration du Runner

### Prérequis

Le runner doit avoir les tags suivants :
- `build-ratp-apk`

### Services Requis

- `docker:24.0.5-dind` - Docker-in-Docker
- Accès au réseau pour le miroir `mirror.dev.futurmap.net`

### Variables d'Environnement

Aucune variable d'environnement supplémentaire requise (utilise les variables GitLab CI intégrées).

## Monitoring et Debugging

### Logs du Pipeline

Chaque job génère des logs détaillés :
- 🧹 Cleanup process
- 🚀 Build initialization
- 📌 Tag generation
- 🔨 Docker build
- ✅ Success confirmations
- 📱 APK extraction
- 📦 APK verification
- 📋 Metadata generation

### Indicateurs de Succès

- **Build réussi** : Image Docker créée avec tags multiples
- **APK extraite** : Artefact téléchargeable disponible
- **Métadonnées** : Fichier JSON généré avec informations complètes

### Gestion d'Erreurs

- **Cleanup failure** : N'arrête pas le pipeline (`allow_failure: true`)
- **Build failure** : Arrête le pipeline, logs détaillés disponibles
- **APK extraction failure** : Arrête le pipeline, vérification des chemins

## Utilisation des Artefacts

### Téléchargement via Interface GitLab

1. Aller dans **CI/CD > Pipelines**
2. Sélectionner le pipeline souhaité
3. Cliquer sur **Download** dans la section **Job artifacts**
4. Télécharger l'archive contenant l'APK et les métadonnées
r
### Téléchargement via API

```bash
# Récupérer l'APK du dernier pipeline réussi
curl --header "PRIVATE-TOKEN: $GITLAB_TOKEN" \
  "https://gitlab.com/api/v4/projects/$PROJECT_ID/jobs/artifacts/main/download?job=extract_apk" \
  --output mergin-maps-apk.zip
```

## Déploiement Optionnel

Un stage de déploiement vers le registry GitLab est disponible (commenté) :

```yaml
deploy_to_registry:
  stage: deploy
  # ... configuration ...
  rules:
    - if: $CI_COMMIT_REF_NAME == "main" || $CI_COMMIT_REF_NAME == "master"
  when: manual
```

Pour l'activer :
1. Décommenter le job `deploy_to_registry`
2. Configurer les variables de registry GitLab
3. Le déploiement sera manuel sur les branches main/master

## Maintenance

### Mise à Jour des Images Docker

Pour mettre à jour les images Docker utilisées :
```yaml
image: docker:24.0.5  # Mettre à jour vers la dernière version
services:
  - docker:24.0.5-dind  # Mettre à jour vers la dernière version
```

### Nettoyage des Artefacts

Les artefacts expirent automatiquement après 30 jours. Pour les conserver plus longtemps, modifier :
```yaml
artifacts:
  expire_in: 30 days  # Changer la durée selon les besoins
```

## Support

Pour toute question ou problème :
1. Vérifier les logs du pipeline dans GitLab
2. Consulter la documentation GitLab CI/CD
3. Vérifier la configuration du runner
4. Contactez l'équipe DevOps

---

**Note** : Ce pipeline remplace complètement l'ancien système de build en deux étapes. Il utilise maintenant le nouveau Dockerfile multi-stage pour une construction plus efficace et standardisée.
