# Mergin Maps Mobile - Build avec Podman

## Vue d'ensemble

Ce projet utilise Podman pour construire l'application Mergin Maps Mobile avec une approche optimisée et sécurisée. Toutes les dépendances externes sont gérées via des variables GitLab pointant vers un Artifactory interne.

## Architecture

### Dockerfile unique optimisé
- **Multi-stage build** : Base Ubuntu + Qt, puis build final
- **Variables d'environnement** : Toutes les URLs externes sont paramétrables
- **Optimisation** : Nettoyage des artefacts de build pour réduire la taille
- **Compatibilité Podman** : Optimisé pour l'exécution rootless

### Pipeline GitLab CI/CD
- **3 stages** : cleanup, podman_build, apk_export
- **Cache intelligent** : Réutilisation des layers entre builds
- **Variables Artifactory** : Centralisation des dépendances externes
- **Sécurité** : Exécution rootless avec Podman

## Variables d'environnement

### Variables GitLab CI/CD
```yaml
# Configuration Podman
PODMAN_HOST: "unix:///run/user/$(id -u)/podman/podman.sock"
CONTAINER_RUNTIME: "podman"

# URLs Artifactory
ARTIFACTORY_URL: "https://artifactory.company.com"
ANDROID_SDK_URL: "${ARTIFACTORY_URL}/android-sdk"
MERGIN_SDK_ARM_URL: "${ARTIFACTORY_URL}/mergin-sdk/arm-android-20240517-224/..."
MERGIN_SDK_ARM64_URL: "${ARTIFACTORY_URL}/mergin-sdk/arm64-android-20240517-224/..."
SDKMAN_URL: "${ARTIFACTORY_URL}/sdkman/get.sdkman.io"
SDKMAN_API_URL: "${ARTIFACTORY_URL}/sdkman/api/2"
ZIP_URL: "$QT_ZIP_URL"         # Qt dependencies ZIP in Artifactory
QT_VERSION: "6.6.3"            # Optional override
```

### Variables locales
```bash
# Pour le build local
export ARTIFACTORY_URL="https://artifactory.company.com"
export SDKMAN_API_URL="https://artifactory.company.com/sdkman/api/2"
export VERSION="1.0.0"  # Pour les builds versionnés
```

## Utilisation

### Build local
```bash
# Build avec tag 'latest'
./build.sh latest

# Build avec version
VERSION=1.0.0 ./build.sh version

# Build avec date
./build.sh date

# Build avec commit git
./build.sh git

# Build combiné
VERSION=1.0.0 ./build.sh combined
```

### Build manuel avec Podman
```bash
# Build simple
podman build \
  --add-host="mirror.dev.futurmap.net:192.168.0.109" \
  --network=host \
  --build-arg ZIP_URL="$QT_ZIP_URL" \
  --build-arg QT_VERSION="6.6.3" \
  -t mergin-maps:latest .

# Build avec variables Artifactory
podman build \
  --add-host="mirror.dev.futurmap.net:192.168.0.109" \
  --network=host \
  --build-arg ANDROID_SDK_URL="https://artifactory.company.com/android-sdk" \
  --build-arg MERGIN_SDK_ARM_URL="https://artifactory.company.com/mergin-sdk/..." \
  --build-arg MERGIN_SDK_ARM64_URL="https://artifactory.company.com/mergin-sdk/..." \
  --build-arg SDKMAN_URL="https://artifactory.company.com/sdkman/get.sdkman.io" \
  --build-arg SDKMAN_API_URL="https://artifactory.company.com/sdkman/api/2" \
  --build-arg ZIP_URL="$QT_ZIP_URL" \
  --build-arg QT_VERSION="6.6.3" \
  -t mergin-maps:latest .
```

## Qt via ZIP (reference)

"Documentation (with examples) in English"

The base image expects a Qt ZIP layout like:

```text
deps/qt/${QT_VERSION}/gcc_64/bin/qmake
deps/qt/${QT_VERSION}/gcc_64/lib/cmake/Qt6/
deps/qt/${QT_VERSION}/android_arm64_v8a/lib/cmake/Qt6/qt.toolchain.cmake
```

### Extraction de l'APK
```bash
# Créer le dossier de sortie
mkdir -p output

# Extraire l'APK
podman run --rm \
  -v "$(pwd)/output:/output:Z" \
  mergin-maps:latest \
  cp /var/www/mergin-maps/build/app/android-build/build/outputs/apk/release/android-build-release.apk /output/
```

## Configuration GitLab Runner

### Prérequis
- Podman installé sur le runner
- Configuration rootless activée
- Accès au réseau pour Artifactory

### Configuration runner
```toml
[[runners]]
  name = "podman-runner"
  executor = "shell"
  [runners.custom_build_dir]
  [runners.cache]
  [runners.custom]
    prepare_exec = "podman unshare --help"
```

## Avantages de l'approche

### 1. **Sécurité**
- Exécution rootless avec Podman
- Pas de démon centralisé
- Isolation des processus

### 2. **Performance**
- Cache intelligent des layers
- Build multi-stage optimisé
- Réduction de la taille d'image

### 3. **Maintenance**
- Variables centralisées pour les dépendances
- Configuration Artifactory unique
- Pipeline simplifié

### 4. **Flexibilité**
- Support de différents tags
- Build local et CI/CD
- Variables d'environnement configurables

## Dépannage

### Problèmes courants

1. **Erreur de permissions Podman**
   ```bash
   podman unshare --help
   ```

2. **Problème de cache**
   ```bash
   podman system prune -f
   podman image prune -f
   ```

3. **Erreur de réseau Artifactory**
   - Vérifier la variable `ARTIFACTORY_URL`
   - Vérifier l'accès au réseau

### Logs utiles

```bash
# Vérifier les images
podman images

# Inspecter une image
podman inspect mergin-maps:latest

# Vérifier les conteneurs
podman ps -a
```

## Métriques

### Temps de build
- **Première fois** : 30-45 minutes
- **Avec cache** : 5-10 minutes
- **Seulement code** : 3-5 minutes

### Taille d'image
- **Finale** : ~2-3 GB (optimisée)
- **Réduction** : ~95% par rapport à l'original

## Exemples d'utilisation

### Build avec Artifactory personnalisé
```bash
# Build avec Artifactory personnalisé
ARTIFACTORY_URL="https://mon-artifactory.com" ./build.sh latest

# Build avec SDKMAN API personnalisé
SDKMAN_API_URL="https://mon-artifactory.com/sdkman/api/2" ./build.sh latest
```

## Support

Pour toute question ou problème :
1. Vérifier les logs GitLab CI/CD
2. Consulter la documentation Podman
3. Vérifier la configuration Artifactory
4. Contacter l'équipe de développement
