/*
 * Copyright (C) 2006 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.content.res;

import android.graphics.Typeface;
import android.util.DisplayMetrics;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.TypedValue;
import org.xmlpull.v1.XmlPullParser;
import android.util.AttributeSet;
import java.io.InputStream;
import android.os.Bundle;
import java.io.IOException;
import org.xmlpull.v1.XmlPullParserException;
import android.content.pm.ActivityInfo;

/**
 * Class for accessing an application's resources.  This sits on top of the
 * asset manager of the application (accessible through {@link #getAssets}) and
 * provides a high-level API for getting typed data from the assets.
 *
 * <p>The Android resource system keeps track of all non-code assets associated with an
 * application. You can use this class to access your application's resources. You can generally
 * acquire the {@link android.content.res.Resources} instance associated with your application
 * with {@link android.content.Context#getResources getResources()}.</p>
 *
 * <p>The Android SDK tools compile your application's resources into the application binary
 * at build time.  To use a resource, you must install it correctly in the source tree (inside
 * your project's {@code res/} directory) and build your application.  As part of the build
 * process, the SDK tools generate symbols for each resource, which you can use in your application
 * code to access the resources.</p>
 *
 * <p>Using application resources makes it easy to update various characteristics of your
 * application without modifying code, and&mdash;by providing sets of alternative
 * resources&mdash;enables you to optimize your application for a variety of device configurations
 * (such as for different languages and screen sizes). This is an important aspect of developing
 * Android applications that are compatible on different types of devices.</p>
 *
 * <p>For more information about using resources, see the documentation about <a
 * href="{@docRoot}guide/topics/resources/index.html">Application Resources</a>.</p>
 * @apiSince 1
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class Resources {

/**
 * Create a new Resources object on top of an existing set of assets in an
 * AssetManager.
 *
 * @deprecated Resources should not be constructed by apps.
 * See {@link android.content.Context#createConfigurationContext(Configuration)}.
 *
 * @param assets Previously created AssetManager.
 * @param metrics Current display metrics to consider when
 *                selecting/computing resource values.
 * @param config Desired device configuration to consider when
 *               selecting/computing resource values (optional).
 * @apiSince 1
 */

@Deprecated
public Resources(android.content.res.AssetManager assets, android.util.DisplayMetrics metrics, android.content.res.Configuration config) { throw new RuntimeException("Stub!"); }

/**
 * Return a global shared Resources object that provides access to only
 * system resources (no application resources), is not configured for the
 * current screen (can not use dimension units, does not change based on
 * orientation, etc), and is not affected by Runtime Resource Overlay.
 * @apiSince 1
 */

public static android.content.res.Resources getSystem() { throw new RuntimeException("Stub!"); }

/**
 * Return the string value associated with a particular resource ID.  The
 * returned object will be a String if this is a plain string; it will be
 * some other type of CharSequence if it is styled.
 * {@more}
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return CharSequence The string data associated with the resource, plus
 *         possibly styled text information.
 
 * This value will never be {@code null}.
 * @apiSince 1
 */

@android.annotation.NonNull
public java.lang.CharSequence getText(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return the Typeface value associated with a particular resource ID.
 * {@more}
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return Typeface The Typeface data associated with the resource.
 
 * This value will never be {@code null}.
 * @apiSince 26
 */

@android.annotation.NonNull
public android.graphics.Typeface getFont(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Returns the character sequence necessary for grammatically correct pluralization
 * of the given resource ID for the given quantity.
 * Note that the character sequence is selected based solely on grammatical necessity,
 * and that such rules differ between languages. Do not assume you know which string
 * will be returned for a given quantity. See
 * <a href="{@docRoot}guide/topics/resources/string-resource.html#Plurals">String Resources</a>
 * for more detail.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 * @param quantity The number used to get the correct string for the current language's
 *           plural rules.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return CharSequence The string data associated with the resource, plus
 *         possibly styled text information.
 
 * This value will never be {@code null}.
 * @apiSince 1
 */

@android.annotation.NonNull
public java.lang.CharSequence getQuantityText(int id, int quantity) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return the string value associated with a particular resource ID.  It
 * will be stripped of any styled text information.
 * {@more}
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return String The string data associated with the resource,
 *         stripped of styled text information.
 
 * This value will never be {@code null}.
 * @apiSince 1
 */

@android.annotation.NonNull
public java.lang.String getString(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return the string value associated with a particular resource ID,
 * substituting the format arguments as defined in {@link java.util.Formatter}
 * and {@link java.lang.String#format}. It will be stripped of any styled text
 * information.
 * {@more}
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @param formatArgs The format arguments that will be used for substitution.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return String The string data associated with the resource,
 *         stripped of styled text information.
 
 * This value will never be {@code null}.
 * @apiSince 1
 */

@android.annotation.NonNull
public java.lang.String getString(int id, java.lang.Object... formatArgs) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Formats the string necessary for grammatically correct pluralization
 * of the given resource ID for the given quantity, using the given arguments.
 * Note that the string is selected based solely on grammatical necessity,
 * and that such rules differ between languages. Do not assume you know which string
 * will be returned for a given quantity. See
 * <a href="{@docRoot}guide/topics/resources/string-resource.html#Plurals">String Resources</a>
 * for more detail.
 *
 * <p>Substitution of format arguments works as if using
 * {@link java.util.Formatter} and {@link java.lang.String#format}.
 * The resulting string will be stripped of any styled text information.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 * @param quantity The number used to get the correct string for the current language's
 *           plural rules.
 * @param formatArgs The format arguments that will be used for substitution.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return String The string data associated with the resource,
 * stripped of styled text information.
 
 * This value will never be {@code null}.
 * @apiSince 1
 */

@android.annotation.NonNull
public java.lang.String getQuantityString(int id, int quantity, java.lang.Object... formatArgs) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Returns the string necessary for grammatically correct pluralization
 * of the given resource ID for the given quantity.
 * Note that the string is selected based solely on grammatical necessity,
 * and that such rules differ between languages. Do not assume you know which string
 * will be returned for a given quantity. See
 * <a href="{@docRoot}guide/topics/resources/string-resource.html#Plurals">String Resources</a>
 * for more detail.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 * @param quantity The number used to get the correct string for the current language's
 *           plural rules.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return String The string data associated with the resource,
 * stripped of styled text information.
 
 * This value will never be {@code null}.
 * @apiSince 1
 */

@android.annotation.NonNull
public java.lang.String getQuantityString(int id, int quantity) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return the string value associated with a particular resource ID.  The
 * returned object will be a String if this is a plain string; it will be
 * some other type of CharSequence if it is styled.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @param def The default CharSequence to return.
 *
 * @return CharSequence The string data associated with the resource, plus
 *         possibly styled text information, or def if id is 0 or not found.
 * @apiSince 1
 */

public java.lang.CharSequence getText(int id, java.lang.CharSequence def) { throw new RuntimeException("Stub!"); }

/**
 * Return the styled text array associated with a particular resource ID.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return The styled text array associated with the resource.
 
 * This value will never be {@code null}.
 * @apiSince 1
 */

@android.annotation.NonNull
public java.lang.CharSequence[] getTextArray(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return the string array associated with a particular resource ID.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return The string array associated with the resource.
 
 * This value will never be {@code null}.
 * @apiSince 1
 */

@android.annotation.NonNull
public java.lang.String[] getStringArray(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return the int array associated with a particular resource ID.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return The int array associated with the resource.
 
 * This value will never be {@code null}.
 * @apiSince 1
 */

@android.annotation.NonNull
public int[] getIntArray(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return an array of heterogeneous values.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return Returns a TypedArray holding an array of the array values.
 * Be sure to call {@link TypedArray#recycle() TypedArray.recycle()}
 * when done with it.
 
 * This value will never be {@code null}.
 * @apiSince 1
 */

@android.annotation.NonNull
public android.content.res.TypedArray obtainTypedArray(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Retrieve a dimensional for a particular resource ID.  Unit
 * conversions are based on the current {@link DisplayMetrics} associated
 * with the resources.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @return Resource dimension value multiplied by the appropriate metric to convert to pixels.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @see #getDimensionPixelOffset
 * @see #getDimensionPixelSize
 * @apiSince 1
 */

public float getDimension(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Retrieve a dimensional for a particular resource ID for use
 * as an offset in raw pixels.  This is the same as
 * {@link #getDimension}, except the returned value is converted to
 * integer pixels for you.  An offset conversion involves simply
 * truncating the base value to an integer.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @return Resource dimension value multiplied by the appropriate
 * metric and truncated to integer pixels.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @see #getDimension
 * @see #getDimensionPixelSize
 * @apiSince 1
 */

public int getDimensionPixelOffset(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Retrieve a dimensional for a particular resource ID for use
 * as a size in raw pixels.  This is the same as
 * {@link #getDimension}, except the returned value is converted to
 * integer pixels for use as a size.  A size conversion involves
 * rounding the base value, and ensuring that a non-zero base value
 * is at least one pixel in size.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @return Resource dimension value multiplied by the appropriate
 * metric and truncated to integer pixels.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @see #getDimension
 * @see #getDimensionPixelOffset
 * @apiSince 1
 */

public int getDimensionPixelSize(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Retrieve a fractional unit for a particular resource ID.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 * @param base The base value of this fraction.  In other words, a
 *             standard fraction is multiplied by this value.
 * @param pbase The parent base value of this fraction.  In other
 *             words, a parent fraction (nn%p) is multiplied by this
 *             value.
 *
 * @return Attribute fractional value multiplied by the appropriate
 * base value.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 * @apiSince 3
 */

public float getFraction(int id, int base, int pbase) { throw new RuntimeException("Stub!"); }

/**
 * Return a drawable object associated with a particular resource ID.
 * Various types of objects will be returned depending on the underlying
 * resource -- for example, a solid color, PNG image, scalable image, etc.
 * The Drawable API hides these implementation details.
 *
 * <p class="note"><strong>Note:</strong> Prior to
 * {@link android.os.Build.VERSION_CODES#JELLY_BEAN}, this function
 * would not correctly retrieve the final configuration density when
 * the resource ID passed here is an alias to another Drawable resource.
 * This means that if the density configuration of the alias resource
 * is different than the actual resource, the density of the returned
 * Drawable would be incorrect, resulting in bad scaling. To work
 * around this, you can instead manually resolve the aliased reference
 * by using {@link #getValue(int, TypedValue, boolean)} and passing
 * {@code true} for {@code resolveRefs}. The resulting
 * {@link TypedValue#resourceId} value may be passed to this method.</p>
 *
 * <p class="note"><strong>Note:</strong> To obtain a themed drawable, use
 * {@link android.content.Context#getDrawable(int) Context.getDrawable(int)}
 * or {@link #getDrawable(int, Theme)} passing the desired theme.</p>
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 * @return Drawable An object that can be used to draw this resource.
 * @throws NotFoundException Throws NotFoundException if the given ID does
 *         not exist.
 * @see #getDrawable(int, Theme)
 * @deprecated Use {@link #getDrawable(int, Theme)} instead.
 * @apiSince 1
 * @deprecatedSince 22
 */

@Deprecated
public android.graphics.drawable.Drawable getDrawable(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return a drawable object associated with a particular resource ID and
 * styled for the specified theme. Various types of objects will be
 * returned depending on the underlying resource -- for example, a solid
 * color, PNG image, scalable image, etc.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 * @param theme The theme used to style the drawable attributes, may be {@code null}.
 * This value may be {@code null}.
 * @return Drawable An object that can be used to draw this resource.
 * @throws NotFoundException Throws NotFoundException if the given ID does
 *         not exist.
 * @apiSince 21
 */

public android.graphics.drawable.Drawable getDrawable(int id, @android.annotation.Nullable android.content.res.Resources.Theme theme) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return a drawable object associated with a particular resource ID for the
 * given screen density in DPI. This will set the drawable's density to be
 * the device's density multiplied by the ratio of actual drawable density
 * to requested density. This allows the drawable to be scaled up to the
 * correct size if needed. Various types of objects will be returned
 * depending on the underlying resource -- for example, a solid color, PNG
 * image, scalable image, etc. The Drawable API hides these implementation
 * details.
 *
 * <p class="note"><strong>Note:</strong> To obtain a themed drawable, use
 * {@link android.content.Context#getDrawable(int) Context.getDrawable(int)}
 * or {@link #getDrawableForDensity(int, int, Theme)} passing the desired
 * theme.</p>
 *
 * @param id The desired resource identifier, as generated by the aapt tool.
 *            This integer encodes the package, type, and resource entry.
 *            The value 0 is an invalid identifier.
 * @param density the desired screen density indicated by the resource as
 *            found in {@link DisplayMetrics}. A value of 0 means to use the
 *            density returned from {@link #getConfiguration()}.
 *            This is equivalent to calling {@link #getDrawable(int)}.
 * @return Drawable An object that can be used to draw this resource.
 * This value may be {@code null}.
 * @throws NotFoundException Throws NotFoundException if the given ID does
 *             not exist.
 * @see #getDrawableForDensity(int, int, Theme)
 * @deprecated Use {@link #getDrawableForDensity(int, int, Theme)} instead.
 * @apiSince 15
 * @deprecatedSince 22
 */

@Deprecated
@android.annotation.Nullable
public android.graphics.drawable.Drawable getDrawableForDensity(int id, int density) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return a drawable object associated with a particular resource ID for the
 * given screen density in DPI and styled for the specified theme.
 *
 * @param id The desired resource identifier, as generated by the aapt tool.
 *            This integer encodes the package, type, and resource entry.
 *            The value 0 is an invalid identifier.
 * @param density The desired screen density indicated by the resource as
 *            found in {@link DisplayMetrics}. A value of 0 means to use the
 *            density returned from {@link #getConfiguration()}.
 *            This is equivalent to calling {@link #getDrawable(int, Theme)}.
 * @param theme The theme used to style the drawable attributes, may be {@code null} if the
 *              drawable cannot be decoded.
 * This value may be {@code null}.
 * @return Drawable An object that can be used to draw this resource.
 * @throws NotFoundException Throws NotFoundException if the given ID does
 *             not exist.
 * @apiSince 21
 */

@android.annotation.Nullable
public android.graphics.drawable.Drawable getDrawableForDensity(int id, int density, @android.annotation.Nullable android.content.res.Resources.Theme theme) { throw new RuntimeException("Stub!"); }

/**
 * Return a movie object associated with the particular resource ID.
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @deprecated Prefer {@link android.graphics.drawable.AnimatedImageDrawable}.
 * @apiSince 1
 * @deprecatedSince 29
 */

@Deprecated
public android.graphics.Movie getMovie(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Returns a color integer associated with a particular resource ID. If the
 * resource holds a complex {@link ColorStateList}, then the default color
 * from the set is returned.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does
 *         not exist.
 *
 * @return A single color value in the form 0xAARRGGBB.
 * @deprecated Use {@link #getColor(int, Theme)} instead.
 * @apiSince 1
 * @deprecatedSince 23
 */

@Deprecated
public int getColor(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Returns a themed color integer associated with a particular resource ID.
 * If the resource holds a complex {@link ColorStateList}, then the default
 * color from the set is returned.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 * @param theme The theme used to style the color attributes, may be
 *              {@code null}.
 *
 * This value may be {@code null}.
 * @throws NotFoundException Throws NotFoundException if the given ID does
 *         not exist.
 *
 * @return A single color value in the form 0xAARRGGBB.
 * @apiSince 23
 */

public int getColor(int id, @android.annotation.Nullable android.content.res.Resources.Theme theme) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Returns a color state list associated with a particular resource ID. The
 * resource may contain either a single raw color value or a complex
 * {@link ColorStateList} holding multiple possible colors.
 *
 * @param id The desired resource identifier of a {@link ColorStateList},
 *           as generated by the aapt tool. This integer encodes the
 *           package, type, and resource entry. The value 0 is an invalid
 *           identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does
 *         not exist.
 *
 * @return A ColorStateList object containing either a single solid color
 *         or multiple colors that can be selected based on a state.
 * This value will never be {@code null}.
 * @deprecated Use {@link #getColorStateList(int, Theme)} instead.
 * @apiSince 1
 * @deprecatedSince 23
 */

@Deprecated
@android.annotation.NonNull
public android.content.res.ColorStateList getColorStateList(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Returns a themed color state list associated with a particular resource
 * ID. The resource may contain either a single raw color value or a
 * complex {@link ColorStateList} holding multiple possible colors.
 *
 * @param id The desired resource identifier of a {@link ColorStateList},
 *           as generated by the aapt tool. This integer encodes the
 *           package, type, and resource entry. The value 0 is an invalid
 *           identifier.
 * @param theme The theme used to style the color attributes, may be
 *              {@code null}.
 *
 * This value may be {@code null}.
 * @throws NotFoundException Throws NotFoundException if the given ID does
 *         not exist.
 *
 * @return A themed ColorStateList object containing either a single solid
 *         color or multiple colors that can be selected based on a state.
 * @apiSince 23
 */

@android.annotation.NonNull
public android.content.res.ColorStateList getColorStateList(int id, @android.annotation.Nullable android.content.res.Resources.Theme theme) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return a boolean associated with a particular resource ID.  This can be
 * used with any integral resource value, and will return true if it is
 * non-zero.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return Returns the boolean value contained in the resource.
 * @apiSince 3
 */

public boolean getBoolean(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return an integer associated with a particular resource ID.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return Returns the integer value contained in the resource.
 * @apiSince 1
 */

public int getInteger(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Retrieve a floating-point value for a particular resource ID.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @return Returns the floating-point value contained in the resource.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does
 *         not exist or is not a floating-point value.
 * @apiSince 29
 */

public float getFloat(int id) { throw new RuntimeException("Stub!"); }

/**
 * Return an XmlResourceParser through which you can read a view layout
 * description for the given resource ID.  This parser has limited
 * functionality -- in particular, you can't change its input, and only
 * the high-level events are available.
 *
 * <p>This function is really a simple wrapper for calling
 * {@link #getXml} with a layout resource.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return A new parser object through which you can read
 *         the XML data.
 *
 * This value will never be {@code null}.
 * @see #getXml
 * @apiSince 1
 */

@android.annotation.NonNull
public android.content.res.XmlResourceParser getLayout(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return an XmlResourceParser through which you can read an animation
 * description for the given resource ID.  This parser has limited
 * functionality -- in particular, you can't change its input, and only
 * the high-level events are available.
 *
 * <p>This function is really a simple wrapper for calling
 * {@link #getXml} with an animation resource.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return A new parser object through which you can read
 *         the XML data.
 *
 * This value will never be {@code null}.
 * @see #getXml
 * @apiSince 1
 */

@android.annotation.NonNull
public android.content.res.XmlResourceParser getAnimation(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return an XmlResourceParser through which you can read a generic XML
 * resource for the given resource ID.
 *
 * <p>The XmlPullParser implementation returned here has some limited
 * functionality.  In particular, you can't change its input, and only
 * high-level parsing events are available (since the document was
 * pre-parsed for you at build time, which involved merging text and
 * stripping comments).
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return A new parser object through which you can read
 *         the XML data.
 *
 * This value will never be {@code null}.
 * @see android.util.AttributeSet
 * @apiSince 1
 */

@android.annotation.NonNull
public android.content.res.XmlResourceParser getXml(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Open a data stream for reading a raw resource.  This can only be used
 * with resources whose value is the name of an asset files -- that is, it can be
 * used to open drawable, sound, and raw resources; it will fail on string
 * and color resources.
 *
 * @param id The resource identifier to open, as generated by the aapt tool.
 *
 * @return InputStream Access to the resource data.
 *
 * This value will never be {@code null}.
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 * @apiSince 1
 */

@android.annotation.NonNull
public java.io.InputStream openRawResource(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Open a data stream for reading a raw resource.  This can only be used
 * with resources whose value is the name of an asset file -- that is, it can be
 * used to open drawable, sound, and raw resources; it will fail on string
 * and color resources.
 *
 * @param id The resource identifier to open, as generated by the aapt tool.
 * @param value The TypedValue object to hold the resource information.
 *
 * @return InputStream Access to the resource data.
 *
 * This value will never be {@code null}.
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 * @apiSince 3
 */

@android.annotation.NonNull
public java.io.InputStream openRawResource(int id, android.util.TypedValue value) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Open a file descriptor for reading a raw resource.  This can only be used
 * with resources whose value is the name of an asset files -- that is, it can be
 * used to open drawable, sound, and raw resources; it will fail on string
 * and color resources.
 *
 * <p>This function only works for resources that are stored in the package
 * as uncompressed data, which typically includes things like mp3 files
 * and png images.
 *
 * @param id The resource identifier to open, as generated by the aapt tool.
 *
 * @return AssetFileDescriptor A new file descriptor you can use to read
 * the resource.  This includes the file descriptor itself, as well as the
 * offset and length of data where the resource appears in the file.  A
 * null is returned if the file exists but is compressed.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @apiSince 1
 */

public android.content.res.AssetFileDescriptor openRawResourceFd(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return the raw data associated with a particular resource ID.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 * @param outValue Object in which to place the resource data.
 * @param resolveRefs If true, a resource that is a reference to another
 *                    resource will be followed so that you receive the
 *                    actual final resource data.  If false, the TypedValue
 *                    will be filled in with the reference itself.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @apiSince 1
 */

public void getValue(int id, android.util.TypedValue outValue, boolean resolveRefs) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Get the raw value associated with a resource with associated density.
 *
 * @param id resource identifier
 * @param density density in DPI
 * @param resolveRefs If true, a resource that is a reference to another
 *            resource will be followed so that you receive the actual final
 *            resource data. If false, the TypedValue will be filled in with
 *            the reference itself.
 * @throws NotFoundException Throws NotFoundException if the given ID does
 *             not exist.
 * @see #getValue(String, TypedValue, boolean)
 * @apiSince 15
 */

public void getValueForDensity(int id, int density, android.util.TypedValue outValue, boolean resolveRefs) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return the raw data associated with a particular resource ID.
 * See getIdentifier() for information on how names are mapped to resource
 * IDs, and getString(int) for information on how string resources are
 * retrieved.
 *
 * <p>Note: use of this function is discouraged.  It is much more
 * efficient to retrieve resources by identifier than by name.
 *
 * @param name The name of the desired resource.  This is passed to
 *             getIdentifier() with a default type of "string".
 * @param outValue Object in which to place the resource data.
 * @param resolveRefs If true, a resource that is a reference to another
 *                    resource will be followed so that you receive the
 *                    actual final resource data.  If false, the TypedValue
 *                    will be filled in with the reference itself.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @apiSince 1
 */

public void getValue(java.lang.String name, android.util.TypedValue outValue, boolean resolveRefs) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Returns the resource ID of the resource that was used to create this AttributeSet.
 *
 * @param set AttributeSet for which we want to find the source.
 * This value may be {@code null}.
 * @return The resource ID for the source that is backing the given AttributeSet or
 * {@link Resources#ID_NULL} if the AttributeSet is {@code null}.
 * @apiSince 29
 */

public static int getAttributeSetSourceResId(@android.annotation.Nullable android.util.AttributeSet set) { throw new RuntimeException("Stub!"); }

/**
 * Generate a new Theme object for this set of Resources.  It initially
 * starts out empty.
 *
 * @return Theme The newly created Theme container.
 * @apiSince 1
 */

public final android.content.res.Resources.Theme newTheme() { throw new RuntimeException("Stub!"); }

/**
 * Retrieve a set of basic attribute values from an AttributeSet, not
 * performing styling of them using a theme and/or style resources.
 *
 * @param set The current attribute values to retrieve.
 * @param attrs The specific attributes to be retrieved. These attribute IDs must be sorted in
 *              ascending order.
 * @return Returns a TypedArray holding an array of the attribute values.
 * Be sure to call {@link TypedArray#recycle() TypedArray.recycle()}
 * when done with it.
 *
 * @see Theme#obtainStyledAttributes(AttributeSet, int[], int, int)
 * @apiSince 1
 */

public android.content.res.TypedArray obtainAttributes(android.util.AttributeSet set, int[] attrs) { throw new RuntimeException("Stub!"); }

/**
 * Store the newly updated configuration.
 *
 * @deprecated See {@link android.content.Context#createConfigurationContext(Configuration)}.
 * @apiSince 1
 * @deprecatedSince 25
 */

@Deprecated
public void updateConfiguration(android.content.res.Configuration config, android.util.DisplayMetrics metrics) { throw new RuntimeException("Stub!"); }

/**
 * Return the current display metrics that are in effect for this resource
 * object.  The returned object should be treated as read-only.
 *
 * @return The resource's current display metrics.
 * @apiSince 1
 */

public android.util.DisplayMetrics getDisplayMetrics() { throw new RuntimeException("Stub!"); }

/**
 * Return the current configuration that is in effect for this resource
 * object.  The returned object should be treated as read-only.
 *
 * @return The resource's current configuration.
 * @apiSince 1
 */

public android.content.res.Configuration getConfiguration() { throw new RuntimeException("Stub!"); }

/**
 * Return a resource identifier for the given resource name.  A fully
 * qualified resource name is of the form "package:type/entry".  The first
 * two components (package and type) are optional if defType and
 * defPackage, respectively, are specified here.
 *
 * <p>Note: use of this function is discouraged.  It is much more
 * efficient to retrieve resources by identifier than by name.
 *
 * @param name The name of the desired resource.
 * @param defType Optional default resource type to find, if "type/" is
 *                not included in the name.  Can be null to require an
 *                explicit type.
 * @param defPackage Optional default package to find, if "package:" is
 *                   not included in the name.  Can be null to require an
 *                   explicit package.
 *
 * @return int The associated resource identifier.  Returns 0 if no such
 *         resource was found.  (0 is not a valid resource ID.)
 * @apiSince 1
 */

public int getIdentifier(java.lang.String name, java.lang.String defType, java.lang.String defPackage) { throw new RuntimeException("Stub!"); }

/**
 * Return the full name for a given resource identifier.  This name is
 * a single string of the form "package:type/entry".
 *
 * @param resid The resource identifier whose name is to be retrieved.
 *
 * @return A string holding the name of the resource.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @see #getResourcePackageName
 * @see #getResourceTypeName
 * @see #getResourceEntryName
 * @apiSince 1
 */

public java.lang.String getResourceName(int resid) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return the package name for a given resource identifier.
 *
 * @param resid The resource identifier whose package name is to be
 * retrieved.
 *
 * @return A string holding the package name of the resource.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @see #getResourceName
 * @apiSince 1
 */

public java.lang.String getResourcePackageName(int resid) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return the type name for a given resource identifier.
 *
 * @param resid The resource identifier whose type name is to be
 * retrieved.
 *
 * @return A string holding the type name of the resource.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @see #getResourceName
 * @apiSince 1
 */

public java.lang.String getResourceTypeName(int resid) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return the entry name for a given resource identifier.
 *
 * @param resid The resource identifier whose entry name is to be
 * retrieved.
 *
 * @return A string holding the entry name of the resource.
 *
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @see #getResourceName
 * @apiSince 1
 */

public java.lang.String getResourceEntryName(int resid) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Parse a series of {@link android.R.styleable#Extra &lt;extra&gt;} tags from
 * an XML file.  You call this when you are at the parent tag of the
 * extra tags, and it will return once all of the child tags have been parsed.
 * This will call {@link #parseBundleExtra} for each extra tag encountered.
 *
 * @param parser The parser from which to retrieve the extras.
 * @param outBundle A Bundle in which to place all parsed extras.
 * @throws XmlPullParserException
 * @throws IOException
 * @apiSince 3
 */

public void parseBundleExtras(android.content.res.XmlResourceParser parser, android.os.Bundle outBundle) throws java.io.IOException, org.xmlpull.v1.XmlPullParserException { throw new RuntimeException("Stub!"); }

/**
 * Parse a name/value pair out of an XML tag holding that data.  The
 * AttributeSet must be holding the data defined by
 * {@link android.R.styleable#Extra}.  The following value types are supported:
 * <ul>
 * <li> {@link TypedValue#TYPE_STRING}:
 * {@link Bundle#putCharSequence Bundle.putCharSequence()}
 * <li> {@link TypedValue#TYPE_INT_BOOLEAN}:
 * {@link Bundle#putCharSequence Bundle.putBoolean()}
 * <li> {@link TypedValue#TYPE_FIRST_INT}-{@link TypedValue#TYPE_LAST_INT}:
 * {@link Bundle#putCharSequence Bundle.putBoolean()}
 * <li> {@link TypedValue#TYPE_FLOAT}:
 * {@link Bundle#putCharSequence Bundle.putFloat()}
 * </ul>
 *
 * @param tagName The name of the tag these attributes come from; this is
 * only used for reporting error messages.
 * @param attrs The attributes from which to retrieve the name/value pair.
 * @param outBundle The Bundle in which to place the parsed value.
 * @throws XmlPullParserException If the attributes are not valid.
 * @apiSince 3
 */

public void parseBundleExtra(java.lang.String tagName, android.util.AttributeSet attrs, android.os.Bundle outBundle) throws org.xmlpull.v1.XmlPullParserException { throw new RuntimeException("Stub!"); }

/**
 * Retrieve underlying AssetManager storage for these resources.
 * @apiSince 1
 */

public final android.content.res.AssetManager getAssets() { throw new RuntimeException("Stub!"); }

/**
 * Call this to remove all cached loaded layout resources from the
 * Resources object.  Only intended for use with performance testing
 * tools.
 * @apiSince 1
 */

public final void flushLayoutCache() { throw new RuntimeException("Stub!"); }

/**
 * Called by zygote when it is done preloading resources, to change back
 * to normal Resources operation.
 * @apiSince 1
 */

public final void finishPreloading() { throw new RuntimeException("Stub!"); }

/**
 * The {@code null} resource ID. This denotes an invalid resource ID that is returned by the
 * system when a resource is not found or the value is set to {@code @null} in XML.
 * @apiSince 29
 */

public static final int ID_NULL = 0; // 0x0
/**
 * This exception is thrown by the resource APIs when a requested resource
 * can not be found.
 * @apiSince 1
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static class NotFoundException extends java.lang.RuntimeException {

/** @apiSince 1 */

public NotFoundException() { throw new RuntimeException("Stub!"); }

/** @apiSince 1 */

public NotFoundException(java.lang.String name) { throw new RuntimeException("Stub!"); }

/** @apiSince 24 */

public NotFoundException(java.lang.String name, java.lang.Exception cause) { throw new RuntimeException("Stub!"); }
}

/**
 * This class holds the current attribute values for a particular theme.
 * In other words, a Theme is a set of values for resource attributes;
 * these are used in conjunction with {@link TypedArray}
 * to resolve the final value for an attribute.
 *
 * <p>The Theme's attributes come into play in two ways: (1) a styled
 * attribute can explicit reference a value in the theme through the
 * "?themeAttribute" syntax; (2) if no value has been defined for a
 * particular styled attribute, as a last resort we will try to find that
 * attribute's value in the Theme.
 *
 * <p>You will normally use the {@link #obtainStyledAttributes} APIs to
 * retrieve XML attributes with style and theme information applied.
 * @apiSince 1
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class Theme {

Theme() { throw new RuntimeException("Stub!"); }

/**
 * Place new attribute values into the theme.  The style resource
 * specified by <var>resid</var> will be retrieved from this Theme's
 * resources, its values placed into the Theme object.
 *
 * <p>The semantics of this function depends on the <var>force</var>
 * argument:  If false, only values that are not already defined in
 * the theme will be copied from the system resource; otherwise, if
 * any of the style's attributes are already defined in the theme, the
 * current values in the theme will be overwritten.
 *
 * @param resId The resource ID of a style resource from which to
 *              obtain attribute values.
 * @param force If true, values in the style resource will always be
 *              used in the theme; otherwise, they will only be used
 *              if not already defined in the theme.
 * @apiSince 1
 */

public void applyStyle(int resId, boolean force) { throw new RuntimeException("Stub!"); }

/**
 * Set this theme to hold the same contents as the theme
 * <var>other</var>.  If both of these themes are from the same
 * Resources object, they will be identical after this function
 * returns.  If they are from different Resources, only the resources
 * they have in common will be set in this theme.
 *
 * @param other The existing Theme to copy from.
 * @apiSince 1
 */

public void setTo(android.content.res.Resources.Theme other) { throw new RuntimeException("Stub!"); }

/**
 * Return a TypedArray holding the values defined by
 * <var>Theme</var> which are listed in <var>attrs</var>.
 *
 * <p>Be sure to call {@link TypedArray#recycle() TypedArray.recycle()} when you are done
 * with the array.
 *
 * @param attrs The desired attributes. These attribute IDs must be sorted in ascending
 *              order.
 *
 * This value must never be {@code null}.
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return Returns a TypedArray holding an array of the attribute values.
 * Be sure to call {@link TypedArray#recycle() TypedArray.recycle()}
 * when done with it.
 *
 * This value will never be {@code null}.
 * @see Resources#obtainAttributes
 * @see #obtainStyledAttributes(int, int[])
 * @see #obtainStyledAttributes(AttributeSet, int[], int, int)
 * @apiSince 1
 */

@androidx.annotation.RecentlyNonNull
public android.content.res.TypedArray obtainStyledAttributes(@androidx.annotation.RecentlyNonNull int[] attrs) { throw new RuntimeException("Stub!"); }

/**
 * Return a TypedArray holding the values defined by the style
 * resource <var>resid</var> which are listed in <var>attrs</var>.
 *
 * <p>Be sure to call {@link TypedArray#recycle() TypedArray.recycle()} when you are done
 * with the array.
 *
 * @param resId The desired style resource.
 * @param attrs The desired attributes in the style. These attribute IDs must be sorted in
 *              ascending order.
 *
 * This value must never be {@code null}.
 * @throws NotFoundException Throws NotFoundException if the given ID does not exist.
 *
 * @return Returns a TypedArray holding an array of the attribute values.
 * Be sure to call {@link TypedArray#recycle() TypedArray.recycle()}
 * when done with it.
 *
 * This value will never be {@code null}.
 * @see Resources#obtainAttributes
 * @see #obtainStyledAttributes(int[])
 * @see #obtainStyledAttributes(AttributeSet, int[], int, int)
 * @apiSince 1
 */

@androidx.annotation.RecentlyNonNull
public android.content.res.TypedArray obtainStyledAttributes(int resId, @androidx.annotation.RecentlyNonNull int[] attrs) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Return a TypedArray holding the attribute values in
 * <var>set</var>
 * that are listed in <var>attrs</var>.  In addition, if the given
 * AttributeSet specifies a style class (through the "style" attribute),
 * that style will be applied on top of the base attributes it defines.
 *
 * <p>Be sure to call {@link TypedArray#recycle() TypedArray.recycle()} when you are done
 * with the array.
 *
 * <p>When determining the final value of a particular attribute, there
 * are four inputs that come into play:</p>
 *
 * <ol>
 *     <li> Any attribute values in the given AttributeSet.
 *     <li> The style resource specified in the AttributeSet (named
 *     "style").
 *     <li> The default style specified by <var>defStyleAttr</var> and
 *     <var>defStyleRes</var>
 *     <li> The base values in this theme.
 * </ol>
 *
 * <p>Each of these inputs is considered in-order, with the first listed
 * taking precedence over the following ones.  In other words, if in the
 * AttributeSet you have supplied <code>&lt;Button
 * textColor="#ff000000"&gt;</code>, then the button's text will
 * <em>always</em> be black, regardless of what is specified in any of
 * the styles.
 *
 * @param set The base set of attribute values.  May be null.
 * This value may be {@code null}.
 * @param attrs The desired attributes to be retrieved. These attribute IDs must be sorted
 *              in ascending order.
 * This value must never be {@code null}.
 * @param defStyleAttr An attribute in the current theme that contains a
 *                     reference to a style resource that supplies
 *                     defaults values for the TypedArray.  Can be
 *                     0 to not look for defaults.
 * @param defStyleRes A resource identifier of a style resource that
 *                    supplies default values for the TypedArray,
 *                    used only if defStyleAttr is 0 or can not be found
 *                    in the theme.  Can be 0 to not look for defaults.
 *
 * @return Returns a TypedArray holding an array of the attribute values.
 * Be sure to call {@link TypedArray#recycle() TypedArray.recycle()}
 * when done with it.
 *
 * @see Resources#obtainAttributes
 * @see #obtainStyledAttributes(int[])
 * @see #obtainStyledAttributes(int, int[])
 * @apiSince 1
 */

@androidx.annotation.RecentlyNonNull
public android.content.res.TypedArray obtainStyledAttributes(@androidx.annotation.RecentlyNullable android.util.AttributeSet set, @androidx.annotation.RecentlyNonNull int[] attrs, int defStyleAttr, int defStyleRes) { throw new RuntimeException("Stub!"); }

/**
 * Retrieve the value of an attribute in the Theme.  The contents of
 * <var>outValue</var> are ultimately filled in by
 * {@link Resources#getValue}.
 *
 * @param resid The resource identifier of the desired theme
 *              attribute.
 * @param outValue Filled in with the ultimate resource value supplied
 *                 by the attribute.
 * @param resolveRefs If true, resource references will be walked; if
 *                    false, <var>outValue</var> may be a
 *                    TYPE_REFERENCE.  In either case, it will never
 *                    be a TYPE_ATTRIBUTE.
 *
 * @return boolean Returns true if the attribute was found and
 *         <var>outValue</var> is valid, else false.
 * @apiSince 1
 */

public boolean resolveAttribute(int resid, android.util.TypedValue outValue, boolean resolveRefs) { throw new RuntimeException("Stub!"); }

/**
 * Returns the resources to which this theme belongs.
 *
 * @return Resources to which this theme belongs.
 * @apiSince 21
 */

public android.content.res.Resources getResources() { throw new RuntimeException("Stub!"); }

/**
 * Return a drawable object associated with a particular resource ID
 * and styled for the Theme.
 *
 * @param id The desired resource identifier, as generated by the aapt
 *           tool. This integer encodes the package, type, and resource
 *           entry. The value 0 is an invalid identifier.
 * @return Drawable An object that can be used to draw this resource.
 * @throws NotFoundException Throws NotFoundException if the given ID
 *         does not exist.
 * @apiSince 21
 */

public android.graphics.drawable.Drawable getDrawable(int id) throws android.content.res.Resources.NotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Returns a bit mask of configuration changes that will impact this
 * theme (and thus require completely reloading it).
 *
 * @return a bit mask of configuration changes, as defined by
 *         {@link ActivityInfo}
 * Value is either <code>0</code> or a combination of {@link android.content.pm.ActivityInfo#CONFIG_MCC}, {@link android.content.pm.ActivityInfo#CONFIG_MNC}, {@link android.content.pm.ActivityInfo#CONFIG_LOCALE}, {@link android.content.pm.ActivityInfo#CONFIG_TOUCHSCREEN}, {@link android.content.pm.ActivityInfo#CONFIG_KEYBOARD}, {@link android.content.pm.ActivityInfo#CONFIG_KEYBOARD_HIDDEN}, {@link android.content.pm.ActivityInfo#CONFIG_NAVIGATION}, {@link android.content.pm.ActivityInfo#CONFIG_ORIENTATION}, {@link android.content.pm.ActivityInfo#CONFIG_SCREEN_LAYOUT}, {@link android.content.pm.ActivityInfo#CONFIG_UI_MODE}, {@link android.content.pm.ActivityInfo#CONFIG_SCREEN_SIZE}, {@link android.content.pm.ActivityInfo#CONFIG_SMALLEST_SCREEN_SIZE}, {@link android.content.pm.ActivityInfo#CONFIG_DENSITY}, {@link android.content.pm.ActivityInfo#CONFIG_LAYOUT_DIRECTION}, {@link android.content.pm.ActivityInfo#CONFIG_COLOR_MODE}, and {@link android.content.pm.ActivityInfo#CONFIG_FONT_SCALE}
 * @see ActivityInfo
 * @apiSince 23
 */

public int getChangingConfigurations() { throw new RuntimeException("Stub!"); }

/**
 * Print contents of this theme out to the log.  For debugging only.
 *
 * @param priority The log priority to use.
 * @param tag The log tag to use.
 * @param prefix Text to prefix each line printed.
 * @apiSince 1
 */

public void dump(int priority, java.lang.String tag, java.lang.String prefix) { throw new RuntimeException("Stub!"); }

/**
 * Rebases the theme against the parent Resource object's current
 * configuration by re-applying the styles passed to
 * {@link #applyStyle(int, boolean)}.
 * @apiSince 29
 */

public void rebase() { throw new RuntimeException("Stub!"); }

/**
 * Returns the resource ID for the style specified using {@code style="..."} in the
 * {@link AttributeSet}'s backing XML element or {@link Resources#ID_NULL} otherwise if not
 * specified or otherwise not applicable.
 * <p>
 * Each {@link android.view.View} can have an explicit style specified in the layout file.
 * This style is used first during the {@link android.view.View} attribute resolution, then
 * if an attribute is not defined there the resource system looks at default style and theme
 * as fallbacks.
 *
 * @param set The base set of attribute values.
 *
 * This value may be {@code null}.
 * @return The resource ID for the style specified using {@code style="..."} in the
 *      {@link AttributeSet}'s backing XML element or {@link Resources#ID_NULL} otherwise
 *      if not specified or otherwise not applicable.
 * @apiSince 29
 */

public int getExplicitStyle(@android.annotation.Nullable android.util.AttributeSet set) { throw new RuntimeException("Stub!"); }

/**
 * Returns the ordered list of resource ID that are considered when resolving attribute
 * values when making an equivalent call to
 * {@link #obtainStyledAttributes(AttributeSet, int[], int, int)} . The list will include
 * a set of explicit styles ({@code explicitStyleRes} and it will include the default styles
 * ({@code defStyleAttr} and {@code defStyleRes}).
 *
 * @param defStyleAttr An attribute in the current theme that contains a
 *                     reference to a style resource that supplies
 *                     defaults values for the TypedArray.  Can be
 *                     0 to not look for defaults.
 * @param defStyleRes A resource identifier of a style resource that
 *                    supplies default values for the TypedArray,
 *                    used only if defStyleAttr is 0 or can not be found
 *                    in the theme.  Can be 0 to not look for defaults.
 * @param explicitStyleRes A resource identifier of an explicit style resource.
 * @return ordered list of resource ID that are considered when resolving attribute values.
 
 * This value will never be {@code null}.
 * @apiSince 29
 */

@android.annotation.NonNull
public int[] getAttributeResolutionStack(int defStyleAttr, int defStyleRes, int explicitStyleRes) { throw new RuntimeException("Stub!"); }
}

}

