/*
 * Copyright (c) 1998, 2013, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package java.security.cert;

import java.util.Iterator;
import java.util.Collection;
import java.security.Security;
import java.security.Provider;
import java.security.NoSuchProviderException;
import java.io.InputStream;
import java.util.List;

/**
 * This class defines the functionality of a certificate factory, which is
 * used to generate certificate, certification path ({@code CertPath})
 * and certificate revocation list (CRL) objects from their encodings.
 *
 * <p>For encodings consisting of multiple certificates, use
 * {@code generateCertificates} when you want to
 * parse a collection of possibly unrelated certificates. Otherwise,
 * use {@code generateCertPath} when you want to generate
 * a {@code CertPath} (a certificate chain) and subsequently
 * validate it with a {@code CertPathValidator}.
 *
 * <p>A certificate factory for X.509 must return certificates that are an
 * instance of {@code java.security.cert.X509Certificate}, and CRLs
 * that are an instance of {@code java.security.cert.X509CRL}.
 *
 * <p>The following example reads a file with Base64 encoded certificates,
 * which are each bounded at the beginning by -----BEGIN CERTIFICATE-----, and
 * bounded at the end by -----END CERTIFICATE-----. We convert the
 * {@code FileInputStream} (which does not support {@code mark}
 * and {@code reset}) to a {@code BufferedInputStream} (which
 * supports those methods), so that each call to
 * {@code generateCertificate} consumes only one certificate, and the
 * read position of the input stream is positioned to the next certificate in
 * the file:
 *
 * <pre>{@code
 * FileInputStream fis = new FileInputStream(filename);
 * BufferedInputStream bis = new BufferedInputStream(fis);
 *
 * CertificateFactory cf = CertificateFactory.getInstance("X.509");
 *
 * while (bis.available() > 0) {
 *    Certificate cert = cf.generateCertificate(bis);
 *    System.out.println(cert.toString());
 * }
 * }</pre>
 *
 * <p>The following example parses a PKCS#7-formatted certificate reply stored
 * in a file and extracts all the certificates from it:
 *
 * <pre>
 * FileInputStream fis = new FileInputStream(filename);
 * CertificateFactory cf = CertificateFactory.getInstance("X.509");
 * Collection c = cf.generateCertificates(fis);
 * Iterator i = c.iterator();
 * while (i.hasNext()) {
 *    Certificate cert = (Certificate)i.next();
 *    System.out.println(cert);
 * }
 * </pre>
 *
 * <p> Android provides the following <code>CertificateFactory</code> types:
 * <table>
 *   <thead>
 *     <tr>
 *       <th>Algorithm</th>
 *       <th>Supported API Levels</th>
 *     </tr>
 *   </thead>
 *   <tbody>
 *     <tr>
 *       <td>X.509</td>
 *       <td>1+</td>
 *     </tr>
 *   </tbody>
 * </table>
 * and the following <code>CertPath</code> encodings:
 * <table>
 *     <thead>
 *         <tr>
 *             <th>Name</th>
 *             <th>Supported (API Levels)</th>
 *         </tr>
 *     </thead>
 *     <tbody>
 *         <tr>
 *             <td>PKCS7</td>
 *             <td>1+</td>
 *         </tr>
 *         <tr>
 *             <td>PkiPath</td>
 *             <td>1+</td>
 *         </tr>
 *     </tbody>
 * </table>
 *
 * The type and encodings are described in the <a href=
 * "{@docRoot}/../technotes/guides/security/StandardNames.html#CertificateFactory">
 * CertificateFactory section</a> and the <a href=
 * "{@docRoot}/../technotes/guides/security/StandardNames.html#CertPathEncodings">
 * CertPath Encodings section</a> of the
 * Java Cryptography Architecture Standard Algorithm Name Documentation.
 *
 * @author Hemma Prafullchandra
 * @author Jan Luehe
 * @author Sean Mullan
 *
 * @see java.security.cert.Certificate
 * @see java.security.cert.X509Certificate
 * @see java.security.cert.CertPath
 * @see java.security.cert.CRL
 * @see java.security.cert.X509CRL
 *
 * @since 1.2
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class CertificateFactory {

/**
 * Creates a CertificateFactory object of the given type, and encapsulates
 * the given provider implementation (SPI object) in it.
 *
 * @param certFacSpi the provider implementation.
 * @param provider the provider.
 * @param type the certificate type.
 */

protected CertificateFactory(java.security.cert.CertificateFactorySpi certFacSpi, java.security.Provider provider, java.lang.String type) { throw new RuntimeException("Stub!"); }

/**
 * Returns a certificate factory object that implements the
 * specified certificate type.
 *
 * <p> This method traverses the list of registered security Providers,
 * starting with the most preferred Provider.
 * A new CertificateFactory object encapsulating the
 * CertificateFactorySpi implementation from the first
 * Provider that supports the specified type is returned.
 *
 * <p> Note that the list of registered providers may be retrieved via
 * the {@link java.security.Security#getProviders() Security.getProviders()} method.
 *
 * @param type the name of the requested certificate type.
 * See the CertificateFactory section in the <a href=
 * "{@docRoot}/../technotes/guides/security/StandardNames.html#CertificateFactory">
 * Java Cryptography Architecture Standard Algorithm Name Documentation</a>
 * for information about standard certificate types.
 *
 * @return a certificate factory object for the specified type.
 *
 * @exception java.security.cert.CertificateException if no Provider supports a
 *          CertificateFactorySpi implementation for the
 *          specified type.
 *
 * @see java.security.Provider
 */

public static final java.security.cert.CertificateFactory getInstance(java.lang.String type) throws java.security.cert.CertificateException { throw new RuntimeException("Stub!"); }

/**
 * Returns a certificate factory object for the specified
 * certificate type.
 *
 * <p> A new CertificateFactory object encapsulating the
 * CertificateFactorySpi implementation from the specified provider
 * is returned.  The specified provider must be registered
 * in the security provider list.
 *
 * <p> Note that the list of registered providers may be retrieved via
 * the {@link java.security.Security#getProviders() Security.getProviders()} method.
 *
 * @param type the certificate type.
 * See the CertificateFactory section in the <a href=
 * "{@docRoot}/../technotes/guides/security/StandardNames.html#CertificateFactory">
 * Java Cryptography Architecture Standard Algorithm Name Documentation</a>
 * for information about standard certificate types.
 *
 * @param provider the name of the provider.
 *
 * @return a certificate factory object for the specified type.
 *
 * @exception java.security.cert.CertificateException if a CertificateFactorySpi
 *          implementation for the specified algorithm is not
 *          available from the specified provider.
 *
 * @exception java.security.NoSuchProviderException if the specified provider is not
 *          registered in the security provider list.
 *
 * @exception java.lang.IllegalArgumentException if the provider name is null
 *          or empty.
 *
 * @see java.security.Provider
 */

public static final java.security.cert.CertificateFactory getInstance(java.lang.String type, java.lang.String provider) throws java.security.cert.CertificateException, java.security.NoSuchProviderException { throw new RuntimeException("Stub!"); }

/**
 * Returns a certificate factory object for the specified
 * certificate type.
 *
 * <p> A new CertificateFactory object encapsulating the
 * CertificateFactorySpi implementation from the specified Provider
 * object is returned.  Note that the specified Provider object
 * does not have to be registered in the provider list.
 *
 * @param type the certificate type.
 * See the CertificateFactory section in the <a href=
 * "{@docRoot}/../technotes/guides/security/StandardNames.html#CertificateFactory">
 * Java Cryptography Architecture Standard Algorithm Name Documentation</a>
 * for information about standard certificate types.
 * @param provider the provider.
 *
 * @return a certificate factory object for the specified type.
 *
 * @exception java.security.cert.CertificateException if a CertificateFactorySpi
 *          implementation for the specified algorithm is not available
 *          from the specified Provider object.
 *
 * @exception java.lang.IllegalArgumentException if the {@code provider} is
 *          null.
 *
 * @see java.security.Provider
 *
 * @since 1.4
 */

public static final java.security.cert.CertificateFactory getInstance(java.lang.String type, java.security.Provider provider) throws java.security.cert.CertificateException { throw new RuntimeException("Stub!"); }

/**
 * Returns the provider of this certificate factory.
 *
 * @return the provider of this certificate factory.
 */

public final java.security.Provider getProvider() { throw new RuntimeException("Stub!"); }

/**
 * Returns the name of the certificate type associated with this
 * certificate factory.
 *
 * @return the name of the certificate type associated with this
 * certificate factory.
 */

public final java.lang.String getType() { throw new RuntimeException("Stub!"); }

/**
 * Generates a certificate object and initializes it with
 * the data read from the input stream {@code inStream}.
 *
 * <p>In order to take advantage of the specialized certificate format
 * supported by this certificate factory,
 * the returned certificate object can be typecast to the corresponding
 * certificate class. For example, if this certificate
 * factory implements X.509 certificates, the returned certificate object
 * can be typecast to the {@code X509Certificate} class.
 *
 * <p>In the case of a certificate factory for X.509 certificates, the
 * certificate provided in {@code inStream} must be DER-encoded and
 * may be supplied in binary or printable (Base64) encoding. If the
 * certificate is provided in Base64 encoding, it must be bounded at
 * the beginning by -----BEGIN CERTIFICATE-----, and must be bounded at
 * the end by -----END CERTIFICATE-----.
 *
 * <p>Note that if the given input stream does not support
 * {@link java.io.InputStream#mark(int) mark} and
 * {@link java.io.InputStream#reset() reset}, this method will
 * consume the entire input stream. Otherwise, each call to this
 * method consumes one certificate and the read position of the
 * input stream is positioned to the next available byte after
 * the inherent end-of-certificate marker. If the data in the input stream
 * does not contain an inherent end-of-certificate marker (other
 * than EOF) and there is trailing data after the certificate is parsed, a
 * {@code CertificateException} is thrown.
 *
 * @param inStream an input stream with the certificate data.
 *
 * @return a certificate object initialized with the data
 * from the input stream.
 *
 * @exception java.security.cert.CertificateException on parsing errors.
 */

public final java.security.cert.Certificate generateCertificate(java.io.InputStream inStream) throws java.security.cert.CertificateException { throw new RuntimeException("Stub!"); }

/**
 * Returns an iteration of the {@code CertPath} encodings supported
 * by this certificate factory, with the default encoding first. See
 * the CertPath Encodings section in the <a href=
 * "{@docRoot}/../technotes/guides/security/StandardNames.html#CertPathEncodings">
 * Java Cryptography Architecture Standard Algorithm Name Documentation</a>
 * for information about standard encoding names and their formats.
 * <p>
 * Attempts to modify the returned {@code Iterator} via its
 * {@code remove} method result in an
 * {@code UnsupportedOperationException}.
 *
 * @return an {@code Iterator} over the names of the supported
 *         {@code CertPath} encodings (as {@code String}s)
 * @since 1.4
 */

public final java.util.Iterator<java.lang.String> getCertPathEncodings() { throw new RuntimeException("Stub!"); }

/**
 * Generates a {@code CertPath} object and initializes it with
 * the data read from the {@code InputStream} inStream. The data
 * is assumed to be in the default encoding. The name of the default
 * encoding is the first element of the {@code Iterator} returned by
 * the {@link #getCertPathEncodings getCertPathEncodings} method.
 *
 * @param inStream an {@code InputStream} containing the data
 * @return a {@code CertPath} initialized with the data from the
 *   {@code InputStream}
 * @exception java.security.cert.CertificateException if an exception occurs while decoding
 * @since 1.4
 */

public final java.security.cert.CertPath generateCertPath(java.io.InputStream inStream) throws java.security.cert.CertificateException { throw new RuntimeException("Stub!"); }

/**
 * Generates a {@code CertPath} object and initializes it with
 * the data read from the {@code InputStream} inStream. The data
 * is assumed to be in the specified encoding. See
 * the CertPath Encodings section in the <a href=
 * "{@docRoot}/../technotes/guides/security/StandardNames.html#CertPathEncodings">
 * Java Cryptography Architecture Standard Algorithm Name Documentation</a>
 * for information about standard encoding names and their formats.
 *
 * @param inStream an {@code InputStream} containing the data
 * @param encoding the encoding used for the data
 * @return a {@code CertPath} initialized with the data from the
 *   {@code InputStream}
 * @exception java.security.cert.CertificateException if an exception occurs while decoding or
 *   the encoding requested is not supported
 * @since 1.4
 */

public final java.security.cert.CertPath generateCertPath(java.io.InputStream inStream, java.lang.String encoding) throws java.security.cert.CertificateException { throw new RuntimeException("Stub!"); }

/**
 * Generates a {@code CertPath} object and initializes it with
 * a {@code List} of {@code Certificate}s.
 * <p>
 * The certificates supplied must be of a type supported by the
 * {@code CertificateFactory}. They will be copied out of the supplied
 * {@code List} object.
 *
 * @param certificates a {@code List} of {@code Certificate}s
 * @return a {@code CertPath} initialized with the supplied list of
 *   certificates
 * @exception java.security.cert.CertificateException if an exception occurs
 * @since 1.4
 */

public final java.security.cert.CertPath generateCertPath(java.util.List<? extends java.security.cert.Certificate> certificates) throws java.security.cert.CertificateException { throw new RuntimeException("Stub!"); }

/**
 * Returns a (possibly empty) collection view of the certificates read
 * from the given input stream {@code inStream}.
 *
 * <p>In order to take advantage of the specialized certificate format
 * supported by this certificate factory, each element in
 * the returned collection view can be typecast to the corresponding
 * certificate class. For example, if this certificate
 * factory implements X.509 certificates, the elements in the returned
 * collection can be typecast to the {@code X509Certificate} class.
 *
 * <p>In the case of a certificate factory for X.509 certificates,
 * {@code inStream} may contain a sequence of DER-encoded certificates
 * in the formats described for
 * {@link #generateCertificate(java.io.InputStream) generateCertificate}.
 * In addition, {@code inStream} may contain a PKCS#7 certificate
 * chain. This is a PKCS#7 <i>SignedData</i> object, with the only
 * significant field being <i>certificates</i>. In particular, the
 * signature and the contents are ignored. This format allows multiple
 * certificates to be downloaded at once. If no certificates are present,
 * an empty collection is returned.
 *
 * <p>Note that if the given input stream does not support
 * {@link java.io.InputStream#mark(int) mark} and
 * {@link java.io.InputStream#reset() reset}, this method will
 * consume the entire input stream.
 *
 * @param inStream the input stream with the certificates.
 *
 * @return a (possibly empty) collection view of
 * java.security.cert.Certificate objects
 * initialized with the data from the input stream.
 *
 * @exception java.security.cert.CertificateException on parsing errors.
 */

public final java.util.Collection<? extends java.security.cert.Certificate> generateCertificates(java.io.InputStream inStream) throws java.security.cert.CertificateException { throw new RuntimeException("Stub!"); }

/**
 * Generates a certificate revocation list (CRL) object and initializes it
 * with the data read from the input stream {@code inStream}.
 *
 * <p>In order to take advantage of the specialized CRL format
 * supported by this certificate factory,
 * the returned CRL object can be typecast to the corresponding
 * CRL class. For example, if this certificate
 * factory implements X.509 CRLs, the returned CRL object
 * can be typecast to the {@code X509CRL} class.
 *
 * <p>Note that if the given input stream does not support
 * {@link java.io.InputStream#mark(int) mark} and
 * {@link java.io.InputStream#reset() reset}, this method will
 * consume the entire input stream. Otherwise, each call to this
 * method consumes one CRL and the read position of the input stream
 * is positioned to the next available byte after the inherent
 * end-of-CRL marker. If the data in the
 * input stream does not contain an inherent end-of-CRL marker (other
 * than EOF) and there is trailing data after the CRL is parsed, a
 * {@code CRLException} is thrown.
 *
 * @param inStream an input stream with the CRL data.
 *
 * @return a CRL object initialized with the data
 * from the input stream.
 *
 * @exception java.security.cert.CRLException on parsing errors.
 */

public final java.security.cert.CRL generateCRL(java.io.InputStream inStream) throws java.security.cert.CRLException { throw new RuntimeException("Stub!"); }

/**
 * Returns a (possibly empty) collection view of the CRLs read
 * from the given input stream {@code inStream}.
 *
 * <p>In order to take advantage of the specialized CRL format
 * supported by this certificate factory, each element in
 * the returned collection view can be typecast to the corresponding
 * CRL class. For example, if this certificate
 * factory implements X.509 CRLs, the elements in the returned
 * collection can be typecast to the {@code X509CRL} class.
 *
 * <p>In the case of a certificate factory for X.509 CRLs,
 * {@code inStream} may contain a sequence of DER-encoded CRLs.
 * In addition, {@code inStream} may contain a PKCS#7 CRL
 * set. This is a PKCS#7 <i>SignedData</i> object, with the only
 * significant field being <i>crls</i>. In particular, the
 * signature and the contents are ignored. This format allows multiple
 * CRLs to be downloaded at once. If no CRLs are present,
 * an empty collection is returned.
 *
 * <p>Note that if the given input stream does not support
 * {@link java.io.InputStream#mark(int) mark} and
 * {@link java.io.InputStream#reset() reset}, this method will
 * consume the entire input stream.
 *
 * @param inStream the input stream with the CRLs.
 *
 * @return a (possibly empty) collection view of
 * java.security.cert.CRL objects initialized with the data from the input
 * stream.
 *
 * @exception java.security.cert.CRLException on parsing errors.
 */

public final java.util.Collection<? extends java.security.cert.CRL> generateCRLs(java.io.InputStream inStream) throws java.security.cert.CRLException { throw new RuntimeException("Stub!"); }
}

