/*
 * Copyright 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.app.appsearch;

import java.util.concurrent.Executor;
import java.util.List;

/**
 * Provides a connection to a single AppSearch database.
 *
 * <p>An {@link android.app.appsearch.AppSearchSession AppSearchSession} instance provides access to database operations such as
 * setting a schema, adding documents, and searching.
 *
 * <p>This class is thread safe.
 *
 * @see android.app.appsearch.GlobalSearchSession
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class AppSearchSession implements java.io.Closeable {

AppSearchSession() { throw new RuntimeException("Stub!"); }

/**
 * Sets the schema that represents the organizational structure of data within the AppSearch
 * database.
 *
 * <p>Upon creating an {@link android.app.appsearch.AppSearchSession AppSearchSession}, {@link #setSchema} should be called. If the
 * schema needs to be updated, or it has not been previously set, then the provided schema will
 * be saved and persisted to disk. Otherwise, {@link #setSchema} is handled efficiently as a
 * no-op call.
 *
 * @param request the schema to set or update the AppSearch database to.
 * This value cannot be {@code null}.
 * @param workExecutor Executor on which to schedule heavy client-side background work such as
 *                     transforming documents.
 * This value cannot be {@code null}.
 * @param callbackExecutor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive errors resulting from setting the schema. If the
 *                 operation succeeds, the callback will be invoked with {@code null}.
 */

public void setSchema(@android.annotation.NonNull android.app.appsearch.SetSchemaRequest request, @android.annotation.NonNull java.util.concurrent.Executor workExecutor, @android.annotation.NonNull java.util.concurrent.Executor callbackExecutor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<android.app.appsearch.SetSchemaResponse>> callback) { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the schema most recently successfully provided to {@link #setSchema}.
 *
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive the pending results of schema.

 * This value cannot be {@code null}.
 */

public void getSchema(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<android.app.appsearch.GetSchemaResponse>> callback) { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the set of all namespaces in the current database with at least one document.
 *
 * @param executor        Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback        Callback to receive the namespaces.

 * This value cannot be {@code null}.
 */

public void getNamespaces(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<java.util.Set<java.lang.String>>> callback) { throw new RuntimeException("Stub!"); }

/**
 * Indexes documents into the {@link android.app.appsearch.AppSearchSession AppSearchSession} database.
 *
 * <p>Each {@link android.app.appsearch.GenericDocument GenericDocument} object must have a {@code schemaType} field set to an {@link android.app.appsearch.AppSearchSchema AppSearchSchema} type that has been previously registered by calling the {@link #setSchema}
 * method.
 *
 * @param request containing documents to be indexed.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive pending result of performing this operation. The keys
 *                 of the returned {@link android.app.appsearch.AppSearchBatchResult AppSearchBatchResult} are the IDs of the input
 *                 documents. The values are {@code null} if they were successfully indexed,
 *                 or a failed {@link android.app.appsearch.AppSearchResult AppSearchResult} otherwise. If an unexpected internal
 *                 error occurs in the AppSearch service,
 *                 {@link android.app.appsearch.BatchResultCallback#onSystemError BatchResultCallback#onSystemError} will be invoked with a
 *                 {@link java.lang.Throwable Throwable}.
 */

public void put(@android.annotation.NonNull android.app.appsearch.PutDocumentsRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.app.appsearch.BatchResultCallback<java.lang.String,java.lang.Void> callback) { throw new RuntimeException("Stub!"); }

/**
 * Gets {@link android.app.appsearch.GenericDocument GenericDocument} objects by document IDs in a namespace from the {@link android.app.appsearch.AppSearchSession AppSearchSession} database.
 *
 * @param request a request containing a namespace and IDs to get documents for.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive the pending result of performing this operation. The keys
 *                 of the returned {@link android.app.appsearch.AppSearchBatchResult AppSearchBatchResult} are the input IDs. The values
 *                 are the returned {@link android.app.appsearch.GenericDocument GenericDocument}s on success, or a failed
 *                 {@link android.app.appsearch.AppSearchResult AppSearchResult} otherwise. IDs that are not found will return a
 *                 failed {@link android.app.appsearch.AppSearchResult AppSearchResult} with a result code of
 *                 {@link android.app.appsearch.AppSearchResult#RESULT_NOT_FOUND AppSearchResult#RESULT_NOT_FOUND}. If an unexpected internal error
 *                 occurs in the AppSearch service, {@link android.app.appsearch.BatchResultCallback#onSystemError BatchResultCallback#onSystemError}
 *                 will be invoked with a {@link java.lang.Throwable Throwable}.

 * This value cannot be {@code null}.
 */

public void getByDocumentId(@android.annotation.NonNull android.app.appsearch.GetByDocumentIdRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.app.appsearch.BatchResultCallback<java.lang.String,android.app.appsearch.GenericDocument> callback) { throw new RuntimeException("Stub!"); }

/**
 * Retrieves documents from the open {@link android.app.appsearch.AppSearchSession AppSearchSession} that match a given query
 * string and type of search provided.
 *
 * <p>Query strings can be empty, contain one term with no operators, or contain multiple terms
 * and operators.
 *
 * <p>For query strings that are empty, all documents that match the {@link android.app.appsearch.SearchSpec SearchSpec} will be
 * returned.
 *
 * <p>For query strings with a single term and no operators, documents that match the provided
 * query string and {@link android.app.appsearch.SearchSpec SearchSpec} will be returned.
 *
 * <p>The following operators are supported:
 *
 * <ul>
 *   <li>AND (implicit)
 *       <p>AND is an operator that matches documents that contain <i>all</i> provided terms.
 *       <p><b>NOTE:</b> A space between terms is treated as an "AND" operator. Explicitly
 *       including "AND" in a query string will treat "AND" as a term, returning documents that
 *       also contain "AND".
 *       <p>Example: "apple AND banana" matches documents that contain the terms "apple", "and",
 *       "banana".
 *       <p>Example: "apple banana" matches documents that contain both "apple" and "banana".
 *       <p>Example: "apple banana cherry" matches documents that contain "apple", "banana", and
 *       "cherry".
 *   <li>OR
 *       <p>OR is an operator that matches documents that contain <i>any</i> provided term.
 *       <p>Example: "apple OR banana" matches documents that contain either "apple" or
 *       "banana".
 *       <p>Example: "apple OR banana OR cherry" matches documents that contain any of "apple",
 *       "banana", or "cherry".
 *   <li>Exclusion (-)
 *       <p>Exclusion (-) is an operator that matches documents that <i>do not</i> contain the
 *       provided term.
 *       <p>Example: "-apple" matches documents that do not contain "apple".
 *   <li>Grouped Terms
 *       <p>For queries that require multiple operators and terms, terms can be grouped into
 *       subqueries. Subqueries are contained within an open "(" and close ")" parenthesis.
 *       <p>Example: "(donut OR bagel) (coffee OR tea)" matches documents that contain either
 *       "donut" or "bagel" and either "coffee" or "tea".
 *   <li>Property Restricts
 *       <p>For queries that require a term to match a specific {@link android.app.appsearch.AppSearchSchema AppSearchSchema} property
 *       of a document, a ":" must be included between the property name and the term.
 *       <p>Example: "subject:important" matches documents that contain the term "important" in
 *       the "subject" property.
 * </ul>
 *
 * <p>Additional search specifications, such as filtering by {@link android.app.appsearch.AppSearchSchema AppSearchSchema} type or
 * adding projection, can be set by calling the corresponding {@link android.app.appsearch.SearchSpec.Builder SearchSpec.Builder} setter.
 *
 * <p>This method is lightweight. The heavy work will be done in {@link android.app.appsearch.SearchResults#getNextPage  }.
 *
 * @param queryExpression query string to search.
 * This value cannot be {@code null}.
 * @param searchSpec spec for setting document filters, adding projection, setting term match
 *     type, etc.
 * This value cannot be {@code null}.
 * @return a {@link android.app.appsearch.SearchResults SearchResults} object for retrieved matched documents.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchResults search(@android.annotation.NonNull java.lang.String queryExpression, @android.annotation.NonNull android.app.appsearch.SearchSpec searchSpec) { throw new RuntimeException("Stub!"); }

/**
 * Retrieves suggested Strings that could be used as {@code queryExpression} in
 * {@link #search(java.lang.String,android.app.appsearch.SearchSpec)} API.
 *
 * <p>The {@code suggestionQueryExpression} can contain one term with no operators, or contain
 * multiple terms and operators. Operators will be considered as a normal term. Please see the
 * operator examples below. The {@code suggestionQueryExpression} must end with a valid term,
 * the suggestions are generated based on the last term. If the input
 * {@code suggestionQueryExpression} doesn't have a valid token, AppSearch will return an
 * empty result list. Please see the invalid examples below.
 *
 * <p>Example: if there are following documents with content stored in AppSearch.
 * <ul>
 *     <li>document1: "term1"
 *     <li>document2: "term1 term2"
 *     <li>document3: "term1 term2 term3"
 *     <li>document4: "org"
 * </ul>
 *
 * <p>Search suggestions with the single term {@code suggestionQueryExpression} "t", the
 * suggested results are:
 * <ul>
 *     <li>"term1" - Use it to be queryExpression in {@link #search} could get 3
 *     {@link android.app.appsearch.SearchResult SearchResult}s, which contains document 1, 2 and 3.
 *     <li>"term2" - Use it to be queryExpression in {@link #search} could get 2
 *     {@link android.app.appsearch.SearchResult SearchResult}s, which contains document 2 and 3.
 *     <li>"term3" - Use it to be queryExpression in {@link #search} could get 1
 *     {@link android.app.appsearch.SearchResult SearchResult}, which contains document 3.
 * </ul>
 *
 * <p>Search suggestions with the multiple term {@code suggestionQueryExpression} "org t", the
 * suggested result will be "org term1" - The last token is completed by the suggested
 * String.
 *
 * <p>Operators in {@link #search} are supported.
 * <p><b>NOTE:</b> Exclusion and Grouped Terms in the last term is not supported.
 * <p>example: "apple -f": This Api will throw an
 * {@link android.app.appsearch.exceptions.AppSearchException} with
 * {@link android.app.appsearch.AppSearchResult#RESULT_INVALID_ARGUMENT AppSearchResult#RESULT_INVALID_ARGUMENT}.
 * <p>example: "apple (f)": This Api will return an empty results.
 *
 * <p>Invalid example: All these input {@code suggestionQueryExpression} don't have a valid
 * last token, AppSearch will return an empty result list.
 * <ul>
 *     <li>""      - Empty {@code suggestionQueryExpression}.
 *     <li>"(f)"   - Ending in a closed brackets.
 *     <li>"f:"    - Ending in an operator.
 *     <li>"f    " - Ending in trailing space.
 * </ul>
 *
 * @param suggestionQueryExpression the non empty query string to search suggestions
 * This value cannot be {@code null}.
 * @param searchSuggestionSpec      spec for setting document filters
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive the pending result of performing this operation, which
 *                 is a List of {@link android.app.appsearch.SearchSuggestionResult SearchSuggestionResult} on success. The returned
 *                 suggestion Strings are ordered by the number of {@link android.app.appsearch.SearchResult SearchResult} you
 *                 could get by using that suggestion in {@link #search}.
 *
 * This value cannot be {@code null}.
 * @see #search(String, SearchSpec)
 */

public void searchSuggestion(@android.annotation.NonNull java.lang.String suggestionQueryExpression, @android.annotation.NonNull android.app.appsearch.SearchSuggestionSpec searchSuggestionSpec, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<java.util.List<android.app.appsearch.SearchSuggestionResult>>> callback) { throw new RuntimeException("Stub!"); }

/**
 * Reports usage of a particular document by namespace and ID.
 *
 * <p>A usage report represents an event in which a user interacted with or viewed a document.
 *
 * <p>For each call to {@link #reportUsage}, AppSearch updates usage count and usage recency
 * metrics for that particular document. These metrics are used for ordering {@link #search}
 * results by the {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_USAGE_COUNT SearchSpec#RANKING_STRATEGY_USAGE_COUNT} and {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_USAGE_LAST_USED_TIMESTAMP  } ranking strategies.
 *
 * <p>Reporting usage of a document is optional.
 *
 * @param request The usage reporting request.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive errors. If the operation succeeds, the callback will be
 *                 invoked with {@code null}.
 */

public void reportUsage(@android.annotation.NonNull android.app.appsearch.ReportUsageRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<java.lang.Void>> callback) { throw new RuntimeException("Stub!"); }

/**
 * Removes {@link android.app.appsearch.GenericDocument GenericDocument} objects by document IDs in a namespace from the {@link android.app.appsearch.AppSearchSession AppSearchSession} database.
 *
 * <p>Removed documents will no longer be surfaced by {@link #search} or {@link
 * #getByDocumentId} calls.
 *
 * <p>Once the database crosses the document count or byte usage threshold, removed documents
 * will be deleted from disk.
 *
 * @param request {@link android.app.appsearch.RemoveByDocumentIdRequest RemoveByDocumentIdRequest} with IDs in a namespace to remove from the
 *     index.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive the pending result of performing this operation. The keys
 *                 of the returned {@link android.app.appsearch.AppSearchBatchResult AppSearchBatchResult} are the input document IDs. The
 *                 values are {@code null} on success, or a failed {@link android.app.appsearch.AppSearchResult AppSearchResult}
 *                 otherwise. IDs that are not found will return a failed
 *                 {@link android.app.appsearch.AppSearchResult AppSearchResult} with a result code of
 *                 {@link android.app.appsearch.AppSearchResult#RESULT_NOT_FOUND AppSearchResult#RESULT_NOT_FOUND}. If an unexpected internal error
 *                 occurs in the AppSearch service, {@link android.app.appsearch.BatchResultCallback#onSystemError BatchResultCallback#onSystemError}
 *                 will be invoked with a {@link java.lang.Throwable Throwable}.
 */

public void remove(@android.annotation.NonNull android.app.appsearch.RemoveByDocumentIdRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.app.appsearch.BatchResultCallback<java.lang.String,java.lang.Void> callback) { throw new RuntimeException("Stub!"); }

/**
 * Removes {@link android.app.appsearch.GenericDocument GenericDocument}s from the index by Query. Documents will be removed if they
 * match the {@code queryExpression} in given namespaces and schemaTypes which is set via {@link android.app.appsearch.SearchSpec.Builder#addFilterNamespaces  } and {@link android.app.appsearch.SearchSpec.Builder#addFilterSchemas SearchSpec.Builder#addFilterSchemas}.
 *
 * <p>An empty {@code queryExpression} matches all documents.
 *
 * <p>An empty set of namespaces or schemaTypes matches all namespaces or schemaTypes in the
 * current database.
 *
 * @param queryExpression Query String to search.
 * This value cannot be {@code null}.
 * @param searchSpec Spec containing schemaTypes, namespaces and query expression indicates how
 *     document will be removed. All specific about how to scoring, ordering, snippeting and
 *     resulting will be ignored.
 * This value cannot be {@code null}.
 * @param executor        Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback        Callback to receive errors resulting from removing the documents. If
 *                        the operation succeeds, the callback will be invoked with
 *                        {@code null}.
 */

public void remove(@android.annotation.NonNull java.lang.String queryExpression, @android.annotation.NonNull android.app.appsearch.SearchSpec searchSpec, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<java.lang.Void>> callback) { throw new RuntimeException("Stub!"); }

/**
 * Gets the storage info for this {@link android.app.appsearch.AppSearchSession AppSearchSession} database.
 *
 * <p>This may take time proportional to the number of documents and may be inefficient to call
 * repeatedly.
 *
 * @param executor        Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback        Callback to receive the storage info.

 * This value cannot be {@code null}.
 */

public void getStorageInfo(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<android.app.appsearch.StorageInfo>> callback) { throw new RuntimeException("Stub!"); }

/**
 * Closes the {@link android.app.appsearch.AppSearchSession AppSearchSession} to persist all schema and document updates,
 * additions, and deletes to disk.
 */

public void close() { throw new RuntimeException("Stub!"); }
}

