/*
 * Copyright (c) 2000, 2017, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package java.nio.channels;

import java.net.SocketOption;
import java.net.ServerSocket;
import java.net.SocketAddress;
import java.nio.channels.spi.SelectorProvider;
import java.io.IOException;

/**
 * A selectable channel for stream-oriented listening sockets.
 *
 * <p> A server-socket channel is created by invoking the {@link #open() open}
 * method of this class.  It is not possible to create a channel for an arbitrary,
 * pre-existing {@link java.net.ServerSocket ServerSocket}. A newly-created server-socket channel is
 * open but not yet bound.  An attempt to invoke the {@link #accept() accept}
 * method of an unbound server-socket channel will cause a {@link java.nio.channels.NotYetBoundException NotYetBoundException}
 * to be thrown. A server-socket channel can be bound by invoking one of the
 * {@link #bind(java.net.SocketAddress,int) bind} methods defined by this class.
 *
 * <p> Socket options are configured using the {@link #setOption(java.net.SocketOption,java.lang.Object)
 * setOption} method. Server-socket channels support the following options:
 * <blockquote>
 * <table class="striped">
 * <caption style="display:none">Socket options</caption>
 * <thead>
 *   <tr>
 *     <th scope="col">Option Name</th>
 *     <th scope="col">Description</th>
 *   </tr>
 * </thead>
 * <tbody>
 *   <tr>
 *     <th scope="row"> {@link java.net.StandardSocketOptions#SO_RCVBUF SO_RCVBUF} </th>
 *     <td> The size of the socket receive buffer </td>
 *   </tr>
 *   <tr>
 *     <th scope="row"> {@link java.net.StandardSocketOptions#SO_REUSEADDR SO_REUSEADDR} </th>
 *     <td> Re-use address </td>
 *   </tr>
 * </tbody>
 * </table>
 * </blockquote>
 * Additional (implementation specific) options may also be supported.
 *
 * <p> Server-socket channels are safe for use by multiple concurrent threads.
 * </p>
 *
 * @author Mark Reinhold
 * @author JSR-51 Expert Group
 * @since 1.4
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract class ServerSocketChannel extends java.nio.channels.spi.AbstractSelectableChannel implements java.nio.channels.NetworkChannel {

/**
 * Initializes a new instance of this class.
 *
 * @param  provider
 *         The provider that created this channel
 */

protected ServerSocketChannel(java.nio.channels.spi.SelectorProvider provider) { super(null); throw new RuntimeException("Stub!"); }

/**
 * Opens a server-socket channel.
 *
 * <p> The new channel is created by invoking the {@link
 * java.nio.channels.spi.SelectorProvider#openServerSocketChannel
 * openServerSocketChannel} method of the system-wide default {@link
 * java.nio.channels.spi.SelectorProvider} object.
 *
 * <p> The new channel's socket is initially unbound; it must be bound to a
 * specific address via one of its socket's {@link
 * java.net.ServerSocket#bind(SocketAddress) bind} methods before
 * connections can be accepted.  </p>
 *
 * @return  A new socket channel
 *
 * @throws  java.io.IOException
 *          If an I/O error occurs
 */

public static java.nio.channels.ServerSocketChannel open() throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Returns an operation set identifying this channel's supported
 * operations.
 *
 * <p> Server-socket channels only support the accepting of new
 * connections, so this method returns {@link java.nio.channels.SelectionKey#OP_ACCEPT SelectionKey#OP_ACCEPT}.
 * </p>
 *
 * @return  The valid-operation set
 */

public final int validOps() { throw new RuntimeException("Stub!"); }

/**
 * Binds the channel's socket to a local address and configures the socket
 * to listen for connections.
 *
 * <p> An invocation of this method is equivalent to the following:
 * <blockquote><pre>
 * bind(local, 0);
 * </pre></blockquote>
 *
 * @param   local
 *          The local address to bind the socket, or {@code null} to bind
 *          to an automatically assigned socket address
 *
 * @return  This channel
 *
 * @throws  java.nio.channels.AlreadyBoundException               {@inheritDoc}
 * @throws  java.nio.channels.UnsupportedAddressTypeException     {@inheritDoc}
 * @throws  java.nio.channels.ClosedChannelException              {@inheritDoc}
 * @throws  java.io.IOException                         {@inheritDoc}
 * @throws  java.lang.SecurityException
 *          If a security manager has been installed and its {@link java.lang.SecurityManager#checkListen checkListen} method denies the
 *          operation
 *
 * @since 1.7
 */

public final java.nio.channels.ServerSocketChannel bind(java.net.SocketAddress local) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Binds the channel's socket to a local address and configures the socket to
 * listen for connections.
 *
 * <p> This method is used to establish an association between the socket and
 * a local address. Once an association is established then the socket remains
 * bound until the channel is closed.
 *
 * <p> The {@code backlog} parameter is the maximum number of pending
 * connections on the socket. Its exact semantics are implementation specific.
 * In particular, an implementation may impose a maximum length or may choose
 * to ignore the parameter altogther. If the {@code backlog} parameter has
 * the value {@code 0}, or a negative value, then an implementation specific
 * default is used.
 *
 * @param   local
 *          The address to bind the socket, or {@code null} to bind to an
 *          automatically assigned socket address
 * @param   backlog
 *          The maximum number of pending connections
 *
 * @return  This channel
 *
 * @throws  java.nio.channels.AlreadyBoundException
 *          If the socket is already bound
 * @throws  java.nio.channels.UnsupportedAddressTypeException
 *          If the type of the given address is not supported
 * @throws  java.nio.channels.ClosedChannelException
 *          If this channel is closed
 * @throws  java.io.IOException
 *          If some other I/O error occurs
 * @throws  java.lang.SecurityException
 *          If a security manager has been installed and its {@link java.lang.SecurityManager#checkListen checkListen} method denies the
 *          operation
 *
 * @since 1.7
 */

public abstract java.nio.channels.ServerSocketChannel bind(java.net.SocketAddress local, int backlog) throws java.io.IOException;

/**
 * @throws  java.lang.UnsupportedOperationException           {@inheritDoc}
 * @throws  java.lang.IllegalArgumentException                {@inheritDoc}
 * @throws  java.nio.channels.ClosedChannelException                  {@inheritDoc}
 * @throws  java.io.IOException                             {@inheritDoc}
 *
 * @since 1.7
 */

public abstract <T> java.nio.channels.ServerSocketChannel setOption(java.net.SocketOption<T> name, T value) throws java.io.IOException;

/**
 * Retrieves a server socket associated with this channel.
 *
 * <p> The returned object will not declare any public methods that are not
 * declared in the {@link java.net.ServerSocket} class.  </p>
 *
 * @return  A server socket associated with this channel
 */

public abstract java.net.ServerSocket socket();

/**
 * Accepts a connection made to this channel's socket.
 *
 * <p> If this channel is in non-blocking mode then this method will
 * immediately return {@code null} if there are no pending connections.
 * Otherwise it will block indefinitely until a new connection is available
 * or an I/O error occurs.
 *
 * <p> The socket channel returned by this method, if any, will be in
 * blocking mode regardless of the blocking mode of this channel.
 *
 * <p> This method performs exactly the same security checks as the {@link
 * java.net.ServerSocket#accept accept} method of the {@link
 * java.net.ServerSocket} class.  That is, if a security manager has been
 * installed then for each new connection this method verifies that the
 * address and port number of the connection's remote endpoint are
 * permitted by the security manager's {@link
 * java.lang.SecurityManager#checkAccept checkAccept} method.  </p>
 *
 * @return  The socket channel for the new connection,
 *          or {@code null} if this channel is in non-blocking mode
 *          and no connection is available to be accepted
 *
 * @throws  java.nio.channels.ClosedChannelException
 *          If this channel is closed
 *
 * @throws  java.nio.channels.AsynchronousCloseException
 *          If another thread closes this channel
 *          while the accept operation is in progress
 *
 * @throws  java.nio.channels.ClosedByInterruptException
 *          If another thread interrupts the current thread
 *          while the accept operation is in progress, thereby
 *          closing the channel and setting the current thread's
 *          interrupt status
 *
 * @throws  java.nio.channels.NotYetBoundException
 *          If this channel's socket has not yet been bound
 *
 * @throws  java.lang.SecurityException
 *          If a security manager has been installed
 *          and it does not permit access to the remote endpoint
 *          of the new connection
 *
 * @throws  java.io.IOException
 *          If some other I/O error occurs
 */

public abstract java.nio.channels.SocketChannel accept() throws java.io.IOException;

/**
 * {@inheritDoc}
 * <p>
 * If there is a security manager set, its {@code checkConnect} method is
 * called with the local address and {@code -1} as its arguments to see
 * if the operation is allowed. If the operation is not allowed,
 * a {@code SocketAddress} representing the
 * {@link java.net.InetAddress#getLoopbackAddress loopback} address and the
 * local port of the channel's socket is returned.
 *
 * @return  The {@code SocketAddress} that the socket is bound to, or the
 *          {@code SocketAddress} representing the loopback address if
 *          denied by the security manager, or {@code null} if the
 *          channel's socket is not bound
 *
 * @throws  java.nio.channels.ClosedChannelException     {@inheritDoc}
 * @throws  java.io.IOException                {@inheritDoc}
 */

public abstract java.net.SocketAddress getLocalAddress() throws java.io.IOException;
}

