/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.health.connect;

import android.health.connect.datatypes.AggregationType;
import android.health.connect.datatypes.Record;
import android.os.OutcomeReceiver;
import java.time.Duration;
import java.time.Period;
import java.time.ZoneOffset;
import java.util.concurrent.Executor;

/**
 * This class provides APIs to interact with the centralized HealthConnect storage maintained by the
 * system.
 *
 * <p>HealthConnect is an offline, on-device storage that unifies data from multiple devices and
 * apps into an ecosystem featuring.
 *
 * <ul>
 *   <li>APIs to insert data of various types into the system.
 * </ul>
 *
 * <p>The basic unit of data in HealthConnect is represented as a {@link android.health.connect.datatypes.Record Record} object, which is
 * the base class for all the other data types such as {@link
 * android.health.connect.datatypes.StepsRecord}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class HealthConnectManager {

HealthConnectManager() { throw new RuntimeException("Stub!"); }

/**
 * Inserts {@code records} into the HealthConnect database. The records returned in {@link android.health.connect.InsertRecordsResponse InsertRecordsResponse} contains the unique IDs of the input records. The values are in same
 * order as {@code records}. In case of an error or a permission failure the HealthConnect
 * service, {@link android.os.OutcomeReceiver#onError OutcomeReceiver#onError} will be invoked with a {@link android.health.connect.HealthConnectException HealthConnectException}.
 *
 * @param records list of records to be inserted.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.RuntimeException for internal errors
 */

public void insertRecords(@android.annotation.NonNull java.util.List<android.health.connect.datatypes.Record> records, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.InsertRecordsResponse,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Get aggregations corresponding to {@code request}.
 *
 * @param <T> Result type of the aggregation.
 *     <p>Note:
 *     <p>This type is embedded in the {@link android.health.connect.datatypes.AggregationType AggregationType} as {@link android.health.connect.datatypes.AggregationType AggregationType} are
 *     typed in nature.
 *     <p>Only {@link android.health.connect.datatypes.AggregationType AggregationType}s that are of same type T can be queried together
 * @param request request for different aggregation.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @see android.health.connect.AggregateRecordsResponse#get
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public <T> void aggregate(@android.annotation.NonNull android.health.connect.AggregateRecordsRequest<T> request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.AggregateRecordsResponse<T>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Get aggregations corresponding to {@code request}. Use this API if results are to be grouped
 * by concrete intervals of time, for example 5 Hrs, 10 Hrs etc.
 *
 * @param <T> Result type of the aggregation.
 *     <p>Note:
 *     <p>This type is embedded in the {@link android.health.connect.datatypes.AggregationType AggregationType} as {@link android.health.connect.datatypes.AggregationType AggregationType} are
 *     typed in nature.
 *     <p>Only {@link android.health.connect.datatypes.AggregationType AggregationType}s that are of same type T can be queried together
 * @param request request for different aggregation.
 * This value cannot be {@code null}.
 * @param duration Duration on which to group by results
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @see android.health.connect.HealthConnectManager#aggregateGroupByPeriod
 */

public <T> void aggregateGroupByDuration(@android.annotation.NonNull android.health.connect.AggregateRecordsRequest<T> request, @android.annotation.NonNull java.time.Duration duration, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.util.List<android.health.connect.AggregateRecordsGroupedByDurationResponse<T>>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Get aggregations corresponding to {@code request}. Use this API if results are to be grouped
 * by number of days. This API handles changes in {@link java.time.ZoneOffset ZoneOffset} when computing the data on
 * a per-day basis.
 *
 * @param <T> Result type of the aggregation.
 *     <p>Note:
 *     <p>This type is embedded in the {@link android.health.connect.datatypes.AggregationType AggregationType} as {@link android.health.connect.datatypes.AggregationType AggregationType} are
 *     typed in nature.
 *     <p>Only {@link android.health.connect.datatypes.AggregationType AggregationType}s that are of same type T can be queried together
 * @param request Request for different aggregation.
 * This value cannot be {@code null}.
 * @param period Period on which to group by results
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @see android.health.connect.AggregateRecordsGroupedByPeriodResponse#get
 * @see android.health.connect.HealthConnectManager#aggregateGroupByDuration
 */

public <T> void aggregateGroupByPeriod(@android.annotation.NonNull android.health.connect.AggregateRecordsRequest<T> request, @android.annotation.NonNull java.time.Period period, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.util.List<android.health.connect.AggregateRecordsGroupedByPeriodResponse<T>>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Deletes records based on {@link android.health.connect.RecordIdFilter RecordIdFilter}.
 *
 * <p>Deletions are performed in a transaction i.e. either all will be deleted or none
 *
 * @param recordIds recordIds on which to perform delete operation.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if {@code recordIds is empty}
 */

public void deleteRecords(@android.annotation.NonNull java.util.List<android.health.connect.RecordIdFilter> recordIds, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.lang.Void,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Deletes records based on the {@link android.health.connect.TimeRangeFilter TimeRangeFilter}.
 *
 * <p>Deletions are performed in a transaction i.e. either all will be deleted or none
 *
 * @param recordType recordType to perform delete operation on.
 * This value cannot be {@code null}.
 * @param timeRangeFilter time filter based on which to delete the records.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.

 * This value cannot be {@code null}.
 */

public void deleteRecords(@android.annotation.NonNull java.lang.Class<? extends android.health.connect.datatypes.Record> recordType, @android.annotation.NonNull android.health.connect.TimeRangeFilter timeRangeFilter, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.lang.Void,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Get change logs post the time when {@code token} was generated.
 *
 * @param changeLogsRequest The token from {@link android.health.connect.HealthConnectManager#getChangeLogToken HealthConnectManager#getChangeLogToken}.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @see android.health.connect.HealthConnectManager#getChangeLogToken
 */

public void getChangeLogs(@android.annotation.NonNull android.health.connect.changelog.ChangeLogsRequest changeLogsRequest, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.changelog.ChangeLogsResponse,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Get token for {HealthConnectManager#getChangeLogs}. Changelogs requested corresponding to
 * this token will be post the time this token was generated by the system all items that match
 * the given filters.
 *
 * <p>Tokens from this request are to be passed to {HealthConnectManager#getChangeLogs}
 *
 * @param request A request to get changelog token
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.

 * This value cannot be {@code null}.
 */

public void getChangeLogToken(@android.annotation.NonNull android.health.connect.changelog.ChangeLogTokenRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.changelog.ChangeLogTokenResponse,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * API to read records based on {@link android.health.connect.ReadRecordsRequestUsingFilters ReadRecordsRequestUsingFilters} or {@link android.health.connect.ReadRecordsRequestUsingIds ReadRecordsRequestUsingIds}
 *
 * <p>Number of records returned by this API will depend based on below factors:
 *
 * <p>When an app with read permission allowed calls the API from background then it will be
 * able to read only its own inserted records and will not get records inserted by other apps.
 * This may be less than the total records present for the record type.
 *
 * <p>When an app with read permission allowed calls the API from foreground then it will be
 * able to read all records for the record type.
 *
 * <p>App with only write permission but no read permission allowed will be able to read only
 * its own inserted records both when in foreground or background.
 *
 * <p>An app without both read and write permissions will not be able to read any record and the
 * API will throw Security Exception.
 *
 * @param request Read request based on {@link android.health.connect.ReadRecordsRequestUsingFilters ReadRecordsRequestUsingFilters} or {@link android.health.connect.ReadRecordsRequestUsingIds ReadRecordsRequestUsingIds}
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if request page size set is more than 5000 in {@link android.health.connect.ReadRecordsRequestUsingFilters ReadRecordsRequestUsingFilters}
 * @throws java.lang.SecurityException if app without read or write permission tries to read.
 */

public <T extends android.health.connect.datatypes.Record> void readRecords(@android.annotation.NonNull android.health.connect.ReadRecordsRequest<T> request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.ReadRecordsResponse<T>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Updates {@code records} into the HealthConnect database. In case of an error or a permission
 * failure the HealthConnect service, {@link android.os.OutcomeReceiver#onError OutcomeReceiver#onError} will be invoked with a
 * {@link android.health.connect.HealthConnectException HealthConnectException}.
 *
 * <p>In case the input record to be updated does not exist in the database or the caller is not
 * the owner of the record then {@link android.health.connect.HealthConnectException#ERROR_INVALID_ARGUMENT HealthConnectException#ERROR_INVALID_ARGUMENT} will be
 * thrown.
 *
 * @param records list of records to be updated.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if at least one of the records is missing both
 *     ClientRecordID and UUID.
 */

public void updateRecords(@android.annotation.NonNull java.util.List<android.health.connect.datatypes.Record> records, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.lang.Void,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Activity action: Launch UI to manage (e.g.&nbsp;grant/revoke) health permissions.
 *
 * <p>Shows a list of apps which request at least one permission of the Health permission group.
 *
 * <p>Input: {@link android.content.Intent#EXTRA_PACKAGE_NAME} string extra with the name of the
 * app requesting the action. Optional: Adding package name extras launches a UI to manager
 * (e.g. grant/revoke) for this app.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.ACTIVITY_INTENT_ACTION) public static final java.lang.String ACTION_MANAGE_HEALTH_PERMISSIONS = "android.health.connect.action.MANAGE_HEALTH_PERMISSIONS";

/**
 * Activity action: Launch UI to share the route associated with an exercise session.
 *
 * <p>Input: caller must provide `String` extra EXTRA_SESSION_ID
 *
 * <p>Result will be delivered via [Activity.onActivityResult] with `ExerciseRoute`
 * EXTRA_EXERCISE_ROUTE.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.ACTIVITY_INTENT_ACTION) public static final java.lang.String ACTION_REQUEST_EXERCISE_ROUTE = "android.health.connect.action.REQUEST_EXERCISE_ROUTE";

/**
 * Used in conjunction with {@link android.content.Intent#ACTION_VIEW_PERMISSION_USAGE} to
 * launch UI to show an app’s health permission rationale/data policy.
 *
 * <p><b>Note:</b> Used by apps to define an intent filter in conjunction with {@link
 * android.content.Intent#ACTION_VIEW_PERMISSION_USAGE} that the HC UI can link out to.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.INTENT_CATEGORY) public static final java.lang.String CATEGORY_HEALTH_PERMISSIONS = "android.intent.category.HEALTH_PERMISSIONS";

/**
 * An exercise route requested via {@link #ACTION_REQUEST_EXERCISE_ROUTE}.
 *
 * <p>This is returned for a successful request to access a route associated with an exercise
 * session.
 */

public static final java.lang.String EXTRA_EXERCISE_ROUTE = "android.health.connect.extra.EXERCISE_ROUTE";

/**
 * A string ID of a session to be used with {@link #ACTION_REQUEST_EXERCISE_ROUTE}.
 *
 * <p>This is used to specify route of which exercise session we want to request.
 */

public static final java.lang.String EXTRA_SESSION_ID = "android.health.connect.extra.SESSION_ID";
}

