/*
 * Copyright (C) 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.net.ipsec.ike;

import android.net.ConnectivityManager;
import android.net.Network;
import android.net.eap.EapSessionConfig;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.Set;

/**
 * IkeSessionParams contains all user provided configurations for negotiating an {@link android.net.ipsec.ike.IkeSession IkeSession}.
 *
 * <p>Note that all negotiated configurations will be reused during rekey including SA Proposal and
 * lifetime.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class IkeSessionParams {

IkeSessionParams() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the configured server hostname
 *
 * <p>The configured server hostname will be resolved during IKE Session creation.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.lang.String getServerHostname() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the configured {@link android.net.Network Network}, or null if was not set.
 *
 * <p>@see {@link android.net.ipsec.ike.IkeSessionParams.Builder#setNetwork(android.net.Network) Builder#setNetwork(Network)}
 */

@android.annotation.Nullable
public android.net.Network getNetwork() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves all IkeSaProposals configured
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<android.net.ipsec.ike.IkeSaProposal> getIkeSaProposals() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the local (client) identity
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeIdentification getLocalIdentification() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the required remote (server) identity
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeIdentification getRemoteIdentification() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the local (client) authentication configuration
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.IkeAuthConfig getLocalAuthConfig() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the remote (server) authentication configuration
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.IkeAuthConfig getRemoteAuthConfig() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves hard lifetime in seconds
 * @return Value is between IKE_HARD_LIFETIME_SEC_MINIMUM and IKE_HARD_LIFETIME_SEC_MAXIMUM inclusive
 */

public int getHardLifetimeSeconds() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves soft lifetime in seconds
 * @return Value is between IKE_SOFT_LIFETIME_SEC_MINIMUM and IKE_HARD_LIFETIME_SEC_MAXIMUM inclusive
 */

public int getSoftLifetimeSeconds() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the Dead Peer Detection(DPD) delay in seconds
 * @return Value is IKE_DPD_DELAY_SEC_MIN or greater
 */

public int getDpdDelaySeconds() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the Network Address Translation Traversal (NATT) keepalive delay in seconds
 * @return Value is between IKE_NATT_KEEPALIVE_DELAY_SEC_MIN and IKE_NATT_KEEPALIVE_DELAY_SEC_MAX inclusive
 */

public int getNattKeepAliveDelaySeconds() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the relative retransmission timeout list in milliseconds
 *
 * <p>@see {@link android.net.ipsec.ike.IkeSessionParams.Builder#setRetransmissionTimeoutsMillis(int[]) Builder#setRetransmissionTimeoutsMillis(int[])}

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public int[] getRetransmissionTimeoutsMillis() { throw new RuntimeException("Stub!"); }

/**
 * Checks if the given IKE Session negotiation option is set
 *
 * @param ikeOption the option to check.
 * Value is {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_ACCEPT_ANY_REMOTE_ID}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_EAP_ONLY_AUTH}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_MOBIKE}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_FORCE_PORT_4500}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_INITIAL_CONTACT}, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_REKEY_MOBILITY, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_ADDRESS_FAMILY_SELECTION, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_NATT_KEEPALIVES, or android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_KEEPALIVE_ON_OFF
 * @throws java.lang.IllegalArgumentException if the provided option is invalid.
 */

public boolean hasIkeOption(int ikeOption) { throw new RuntimeException("Stub!"); }

/**
 * Return all the enabled IKE Options
 *
 * @return A Set of enabled IKE options that have been added using {@link android.net.ipsec.ike.IkeSessionParams.Builder#addIkeOption(int)      }
 
 * This value cannot be {@code null}.

 * Value is {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_ACCEPT_ANY_REMOTE_ID}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_EAP_ONLY_AUTH}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_MOBIKE}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_FORCE_PORT_4500}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_INITIAL_CONTACT}, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_REKEY_MOBILITY, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_ADDRESS_FAMILY_SELECTION, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_NATT_KEEPALIVES, or android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_KEEPALIVE_ON_OFF
 */

@android.annotation.NonNull
public java.util.Set<java.lang.Integer> getIkeOptions() { throw new RuntimeException("Stub!"); }

/** @hide */

public int hashCode() { throw new RuntimeException("Stub!"); }

/** @hide */

public boolean equals(java.lang.Object o) { throw new RuntimeException("Stub!"); }

/**
 * Setting timer to this value will disable the Dead Peer Detection(DPD).
 *
 * <p>@see {@link android.net.ipsec.ike.IkeSessionParams.Builder#setDpdDelaySeconds Builder#setDpdDelaySeconds}
 */

public static final int IKE_DPD_DELAY_SEC_DISABLED = 2147483647; // 0x7fffffff

/**
 * If set, the IKE library will accept any remote (server) identity, even if it does not match
 * the configured remote identity
 *
 * <p>See {@link android.net.ipsec.ike.IkeSessionParams.Builder#setRemoteIdentification(android.net.ipsec.ike.IkeIdentification) Builder#setRemoteIdentification(IkeIdentification)}
 */

public static final int IKE_OPTION_ACCEPT_ANY_REMOTE_ID = 0; // 0x0

/**
 * If set, and EAP has been configured as the authentication method, the IKE library will
 * request that the remote (also) use an EAP-only authentication flow.
 *
 * <p>@see {@link android.net.ipsec.ike.IkeSessionParams.Builder#setAuthEap(java.security.cert.X509Certificate,android.net.eap.EapSessionConfig) Builder#setAuthEap(X509Certificate, EapSessionConfig)}
 */

public static final int IKE_OPTION_EAP_ONLY_AUTH = 1; // 0x1

/**
 * Configures the IKE session to always send to port 4500.
 *
 * <p>If set, the IKE Session will be initiated and maintained exclusively using
 * destination port 4500, regardless of the presence of NAT. Otherwise, the IKE Session will
 * be initiated on destination port 500; then, if either a NAT is detected or both MOBIKE
 * and NAT-T are supported by the peer, it will proceed on port 4500.
 */

public static final int IKE_OPTION_FORCE_PORT_4500 = 3; // 0x3

/**
 * If set, the IKE library will send INITIAL_CONTACT notification to the peers.
 *
 * <p>If this option is set, the INITIAL_CONTACT notification payload is sent in IKE_AUTH. The
 * client can use this option to assert to the peer that this IKE SA is the only IKE SA
 * currently active between the authenticated identities.
 *
 * <p>@see "https://tools.ietf.org/html/rfc7296#section-2.4" RFC 7296, Internet Key Exchange
 * Protocol Version 2 (IKEv2)
 *
 * <p>@see {@link android.net.ipsec.ike.IkeSessionParams.Builder#addIkeOption(int) Builder#addIkeOption(int)}
 */

public static final int IKE_OPTION_INITIAL_CONTACT = 4; // 0x4

/**
 * If set, the IKE Session will attempt to handle IP address changes using RFC4555 MOBIKE.
 *
 * <p>Upon IP address changes (including Network changes), the IKE session will initiate an RFC
 * 4555 MOBIKE procedure, migrating both this IKE Session and associated IPsec Transforms to the
 * new local and remote address pair.
 *
 * <p>The IKE library will first attempt to enable MOBIKE to handle the changes of underlying
 * network and addresses. For callers targeting SDK {@link android.os.Build.VERSION_CODES#S_V2}
 * and earlier, this option will implicitly enable the support for rekey-based mobility, and
 * thus if the server does not support MOBIKE, the IKE Session will try migration by rekeying
 * all associated IPsec SAs. This rekey-based mobility feature is not best-practice and has
 * technical issues; accordingly, it will no longer be enabled for callers targeting SDK {@link
 * android.os.Build.VERSION_CODES#TIRAMISU} and above.
 *
 * <p>Checking whether or not MOBIKE is supported by both the IKE library and the server in an
 * IKE Session is done via {@link android.net.ipsec.ike.IkeSessionConfiguration#isIkeExtensionEnabled(int) IkeSessionConfiguration#isIkeExtensionEnabled(int)}.
 *
 * <p>It is recommended that IKE_OPTION_MOBIKE be enabled unless precluded for compatibility
 * reasons.
 *
 * <p>If this option is set for an IKE Session, Transport-mode SAs will not be allowed in that
 * Session.
 *
 * <p>Callers that need to perform migration of IPsec transforms and tunnels MUST implement
 * migration specific methods in {@link android.net.ipsec.ike.IkeSessionCallback IkeSessionCallback} and {@link android.net.ipsec.ike.ChildSessionCallback ChildSessionCallback}.
 */

public static final int IKE_OPTION_MOBIKE = 2; // 0x2
/** This class can be used to incrementally construct a {@link android.net.ipsec.ike.IkeSessionParams IkeSessionParams}. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class Builder {

/**
 * Construct Builder
 */

public Builder() { throw new RuntimeException("Stub!"); }

/**
 * Construct Builder from the {@link android.net.ipsec.ike.IkeSessionParams IkeSessionParams} object.
 *
 * @param ikeSessionParams the object this Builder will be constructed with.

 * This value cannot be {@code null}.
 */

public Builder(@android.annotation.NonNull android.net.ipsec.ike.IkeSessionParams ikeSessionParams) { throw new RuntimeException("Stub!"); }

/**
 * Sets the server hostname for the {@link android.net.ipsec.ike.IkeSessionParams IkeSessionParams} being built.
 *
 * @param serverHostname the hostname of the IKE server, such as "ike.android.com".
 * This value cannot be {@code null}.
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setServerHostname(@android.annotation.NonNull java.lang.String serverHostname) { throw new RuntimeException("Stub!"); }

/**
 * Sets the {@link android.net.Network Network} for the {@link android.net.ipsec.ike.IkeSessionParams IkeSessionParams} being built.
 *
 * <p>If no {@link android.net.Network Network} is provided, the default Network (as per {@link android.net.ConnectivityManager#getActiveNetwork()  }) will be used when constructing an {@link android.net.ipsec.ike.IkeSession IkeSession}.
 *
 * @param network the {@link android.net.Network Network} that IKE Session will use, or {@code null} to clear
 *     the previously set {@link android.net.Network Network}
 * @return Builder this, to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setNetwork(@android.annotation.Nullable android.net.Network network) { throw new RuntimeException("Stub!"); }

/**
 * Sets local IKE identification for the {@link android.net.ipsec.ike.IkeSessionParams IkeSessionParams} being built.
 *
 * <p>It is not allowed to use KEY ID together with digital-signature-based authentication
 * as per RFC 7296.
 *
 * @param identification the local IKE identification.
 * This value cannot be {@code null}.
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setLocalIdentification(@android.annotation.NonNull android.net.ipsec.ike.IkeIdentification identification) { throw new RuntimeException("Stub!"); }

/**
 * Sets remote IKE identification for the {@link android.net.ipsec.ike.IkeSessionParams IkeSessionParams} being built.
 *
 * @param identification the remote IKE identification.
 * This value cannot be {@code null}.
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setRemoteIdentification(@android.annotation.NonNull android.net.ipsec.ike.IkeIdentification identification) { throw new RuntimeException("Stub!"); }

/**
 * Adds an IKE SA proposal to the {@link android.net.ipsec.ike.IkeSessionParams IkeSessionParams} being built.
 *
 * @param proposal IKE SA proposal.
 * This value cannot be {@code null}.
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder addIkeSaProposal(@android.annotation.NonNull android.net.ipsec.ike.IkeSaProposal proposal) { throw new RuntimeException("Stub!"); }

/**
 * Configures the {@link android.net.ipsec.ike.IkeSession IkeSession} to use pre-shared-key-based authentication.
 *
 * <p>Both client and server MUST be authenticated using the provided shared key. IKE
 * authentication will fail if the remote peer tries to use other authentication methods.
 *
 * <p>Callers MUST declare only one authentication method. Calling this function will
 * override the previously set authentication configuration.
 *
 * <p>Callers SHOULD NOT use this if any other authentication methods can be used; PSK-based
 * authentication is generally considered insecure.
 *
 * @param sharedKey the shared key.
 * This value cannot be {@code null}.
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setAuthPsk(@android.annotation.NonNull byte[] sharedKey) { throw new RuntimeException("Stub!"); }

/**
 * Configures the {@link android.net.ipsec.ike.IkeSession IkeSession} to use EAP authentication.
 *
 * <p>Not all EAP methods provide mutual authentication. As such EAP MUST be used in
 * conjunction with a public-key-signature-based authentication of the remote server, unless
 * EAP-Only authentication is enabled.
 *
 * <p>Callers may enable EAP-Only authentication by setting {@link
 * #IKE_OPTION_EAP_ONLY_AUTH}, which will make IKE library request the remote to use
 * EAP-Only authentication. The remote may opt to reject the request, at which point the
 * received certificates and authentication payload WILL be validated with the provided root
 * CA or system's truststore as usual. Only safe EAP methods as listed in RFC 5998 will be
 * accepted for EAP-Only authentication.
 *
 * <p>If {@link #IKE_OPTION_EAP_ONLY_AUTH} is set, callers MUST configure EAP as the
 * authentication method and all EAP methods set in EAP Session configuration MUST be safe
 * methods that are accepted for EAP-Only authentication. Otherwise callers will get an
 * exception when building the {@link android.net.ipsec.ike.IkeSessionParams IkeSessionParams}
 *
 * <p>Callers MUST declare only one authentication method. Calling this function will
 * override the previously set authentication configuration.
 *
 * @see <a href="https://tools.ietf.org/html/rfc5280">RFC 5280, Internet X.509 Public Key
 *     Infrastructure Certificate and Certificate Revocation List (CRL) Profile</a>
 * @see <a href="https://tools.ietf.org/html/rfc5998">RFC 5998, An Extension for EAP-Only
 *     Authentication in IKEv2</a>
 * @param serverCaCert the CA certificate for validating the received server certificate(s).
 *     If a certificate is provided, it MUST be the root CA used by the server, or
 *     authentication will fail. If no certificate is provided, any root CA in the system's
 *     truststore is considered acceptable.
 * This value may be {@code null}.
 * @param eapConfig This value cannot be {@code null}.
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setAuthEap(@android.annotation.Nullable java.security.cert.X509Certificate serverCaCert, @android.annotation.NonNull android.net.eap.EapSessionConfig eapConfig) { throw new RuntimeException("Stub!"); }

/**
 * Configures the {@link android.net.ipsec.ike.IkeSession IkeSession} to use public-key-signature-based authentication.
 *
 * <p>The public key included by the client end certificate and the private key used for
 * signing MUST be a matching key pair.
 *
 * <p>The IKE library will use the strongest signature algorithm supported by both sides.
 *
 * <p>Currenly only RSA digital signature is supported.
 *
 * @param serverCaCert the CA certificate for validating the received server certificate(s).
 *     If a certificate is provided, it MUST be the root CA used by the server, or
 *     authentication will fail. If no certificate is provided, any root CA in the system's
 *     truststore is considered acceptable.
 * This value may be {@code null}.
 * @param clientEndCert the end certificate for remote server to verify the locally
 *     generated signature.
 * This value cannot be {@code null}.
 * @param clientPrivateKey private key to generate outbound digital signature. The {@link java.security.PrivateKey PrivateKey} MUST be an instance of {@link java.security.interfaces.RSAKey RSAKey}.
 * This value cannot be {@code null}.
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setAuthDigitalSignature(@android.annotation.Nullable java.security.cert.X509Certificate serverCaCert, @android.annotation.NonNull java.security.cert.X509Certificate clientEndCert, @android.annotation.NonNull java.security.PrivateKey clientPrivateKey) { throw new RuntimeException("Stub!"); }

/**
 * Configures the {@link android.net.ipsec.ike.IkeSession IkeSession} to use public-key-signature-based authentication.
 *
 * <p>The public key included by the client end certificate and the private key used for
 * signing MUST be a matching key pair.
 *
 * <p>The IKE library will use the strongest signature algorithm supported by both sides.
 *
 * <p>Currenly only RSA digital signature is supported.
 *
 * @param serverCaCert the CA certificate for validating the received server certificate(s).
 *     If a null value is provided, IKE library will try all default CA certificates stored
 *     in Android system to do the validation. Otherwise, it will only use the provided CA
 *     certificate.
 * @param clientEndCert the end certificate for remote server to verify locally generated
 *     signature.
 * This value cannot be {@code null}.
 * @param clientIntermediateCerts intermediate certificates for the remote server to
 *     validate the end certificate.
 * This value cannot be {@code null}.
 * @param clientPrivateKey private key to generate outbound digital signature. The {@link java.security.PrivateKey PrivateKey} MUST be an instance of {@link java.security.interfaces.RSAKey RSAKey}.
 * This value cannot be {@code null}.
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setAuthDigitalSignature(@android.annotation.Nullable java.security.cert.X509Certificate serverCaCert, @android.annotation.NonNull java.security.cert.X509Certificate clientEndCert, @android.annotation.NonNull java.util.List<java.security.cert.X509Certificate> clientIntermediateCerts, @android.annotation.NonNull java.security.PrivateKey clientPrivateKey) { throw new RuntimeException("Stub!"); }

/**
 * Sets hard and soft lifetimes.
 *
 * <p>Lifetimes will not be negotiated with the remote IKE server.
 *
 * @param hardLifetimeSeconds number of seconds after which IKE SA will expire. Defaults to
 *     14400 seconds (4 hours). MUST be a value from 300 seconds (5 minutes) to 86400
 *     seconds (24 hours), inclusive.
 * Value is between IKE_HARD_LIFETIME_SEC_MINIMUM and IKE_HARD_LIFETIME_SEC_MAXIMUM inclusive
 * @param softLifetimeSeconds number of seconds after which IKE SA will request rekey.
 *     Defaults to 7200 seconds (2 hours). MUST be at least 120 seconds (2 minutes), and at
 *     least 60 seconds (1 minute) shorter than the hard lifetime.
 * Value is between IKE_SOFT_LIFETIME_SEC_MINIMUM and IKE_HARD_LIFETIME_SEC_MAXIMUM inclusive
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setLifetimeSeconds(int hardLifetimeSeconds, int softLifetimeSeconds) { throw new RuntimeException("Stub!"); }

/**
 * Sets the Dead Peer Detection(DPD) delay in seconds.
 *
 * @param dpdDelaySeconds number of seconds after which IKE SA will initiate DPD if no
 *     inbound cryptographically protected IKE message was received. Defaults to 120
 *     seconds. MUST be a value greater than or equal to than 20 seconds. Setting the value
 *     to {@link android.net.ipsec.ike.IkeSessionParams#IKE_DPD_DELAY_SEC_DISABLED IkeSessionParams#IKE_DPD_DELAY_SEC_DISABLED} will disable DPD.
 * Value is IKE_DPD_DELAY_SEC_MIN or greater
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setDpdDelaySeconds(int dpdDelaySeconds) { throw new RuntimeException("Stub!"); }

/**
 * Sets the Network Address Translation Traversal (NATT) keepalive delay in seconds.
 *
 * @param nattKeepaliveDelaySeconds number of seconds between keepalive packet
 *     transmissions. Defaults to 10 seconds. MUST be a value from 10 seconds to 3600
 *     seconds, inclusive.
 * Value is between IKE_NATT_KEEPALIVE_DELAY_SEC_MIN and IKE_NATT_KEEPALIVE_DELAY_SEC_MAX inclusive
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setNattKeepAliveDelaySeconds(int nattKeepaliveDelaySeconds) { throw new RuntimeException("Stub!"); }

/**
 * Sets the retransmission timeout list in milliseconds.
 *
 * <p>Configures the retransmission by providing an array of relative retransmission
 * timeouts in milliseconds. After sending out a request and before receiving the response,
 * the IKE Session will iterate through the array and wait for the relative timeout before
 * the next retry. If the last timeout is exceeded, the IKE Session will be terminated.
 *
 * <p>Each element in the array MUST be a value from 500 ms to 1800000 ms (30 minutes). The
 * length of the array MUST NOT exceed 10. This retransmission timeout list defaults to
 * {0.5s, 1s, 2s, 4s, 8s}
 *
 * @param retransTimeoutMillisList the array of relative retransmission timeout in
 *     milliseconds.
 * This value cannot be {@code null}.
 * @return Builder this, to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder setRetransmissionTimeoutsMillis(@android.annotation.NonNull int[] retransTimeoutMillisList) { throw new RuntimeException("Stub!"); }

/**
 * Sets the specified IKE Option as enabled.
 *
 * @param ikeOption the option to be enabled.
 * Value is {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_ACCEPT_ANY_REMOTE_ID}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_EAP_ONLY_AUTH}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_MOBIKE}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_FORCE_PORT_4500}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_INITIAL_CONTACT}, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_REKEY_MOBILITY, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_ADDRESS_FAMILY_SELECTION, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_NATT_KEEPALIVES, or android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_KEEPALIVE_ON_OFF
 * @return Builder this, to facilitate chaining.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the provided option is invalid.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder addIkeOption(int ikeOption) { throw new RuntimeException("Stub!"); }

/**
 * Resets (disables) the specified IKE Option.
 *
 * @param ikeOption the option to be disabled.
 * Value is {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_ACCEPT_ANY_REMOTE_ID}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_EAP_ONLY_AUTH}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_MOBIKE}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_FORCE_PORT_4500}, {@link android.net.ipsec.ike.IkeSessionParams#IKE_OPTION_INITIAL_CONTACT}, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_REKEY_MOBILITY, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_ADDRESS_FAMILY_SELECTION, android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_NATT_KEEPALIVES, or android.net.ipsec.ike.IkeSessionParams.IKE_OPTION_AUTOMATIC_KEEPALIVE_ON_OFF
 * @return Builder this, to facilitate chaining.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the provided option is invalid.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams.Builder removeIkeOption(int ikeOption) { throw new RuntimeException("Stub!"); }

/**
 * Validates and builds the {@link android.net.ipsec.ike.IkeSessionParams IkeSessionParams}.
 *
 * @return IkeSessionParams the validated IkeSessionParams.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.ipsec.ike.IkeSessionParams build() { throw new RuntimeException("Stub!"); }
}

/** This class contains common information of an IKEv2 authentication configuration. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract static class IkeAuthConfig {

IkeAuthConfig() { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

public boolean equals(java.lang.Object o) { throw new RuntimeException("Stub!"); }
}

/**
 * This class represents the configuration to support IKEv2 public-key-signature-based
 * authentication of the local side.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static class IkeAuthDigitalSignLocalConfig extends android.net.ipsec.ike.IkeSessionParams.IkeAuthConfig {

IkeAuthDigitalSignLocalConfig() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the client end certificate
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.security.cert.X509Certificate getClientEndCertificate() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the intermediate certificates
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<java.security.cert.X509Certificate> getIntermediateCertificates() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the private key
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.security.PrivateKey getPrivateKey() { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

public boolean equals(java.lang.Object o) { throw new RuntimeException("Stub!"); }
}

/**
 * This class represents the configuration to support IKEv2 public-key-signature-based
 * authentication of the remote side.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static class IkeAuthDigitalSignRemoteConfig extends android.net.ipsec.ike.IkeSessionParams.IkeAuthConfig {

IkeAuthDigitalSignRemoteConfig() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the provided CA certificate for validating the remote certificate(s)
 * @return This value may be {@code null}.
 */

@android.annotation.Nullable
public java.security.cert.X509Certificate getRemoteCaCert() { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

public boolean equals(java.lang.Object o) { throw new RuntimeException("Stub!"); }
}

/**
 * This class represents the configuration to support EAP authentication of the local side.
 *
 * <p>@see {@link android.net.ipsec.ike.IkeSessionParams.Builder#setAuthEap(java.security.cert.X509Certificate,android.net.eap.EapSessionConfig) IkeSessionParams.Builder#setAuthEap(X509Certificate, EapSessionConfig)}
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static class IkeAuthEapConfig extends android.net.ipsec.ike.IkeSessionParams.IkeAuthConfig {

IkeAuthEapConfig() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves EAP configuration
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.eap.EapSessionConfig getEapConfig() { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

public boolean equals(java.lang.Object o) { throw new RuntimeException("Stub!"); }
}

/**
 * This class represents the configuration to support IKEv2 pre-shared-key-based authentication
 * of local or remote side.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static class IkeAuthPskConfig extends android.net.ipsec.ike.IkeSessionParams.IkeAuthConfig {

IkeAuthPskConfig() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the pre-shared key
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public byte[] getPsk() { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

public boolean equals(java.lang.Object o) { throw new RuntimeException("Stub!"); }
}

}

