/*
 * Copyright (c) 2020, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package java.util;

import java.io.UncheckedIOException;

/**
 * {@code HexFormat} converts between bytes and chars and hex-encoded strings which may include
 * additional formatting markup such as prefixes, suffixes, and delimiters.
 * <p>
 * There are two factories of {@code HexFormat} with preset parameters {@link #of()} and
 * {@link #ofDelimiter(java.lang.String) ofDelimiter(delimiter)}. For other parameter combinations
 * the {@code withXXX} methods return copies of {@code HexFormat} modified
 * {@link #withPrefix(java.lang.String)}, {@link #withSuffix(java.lang.String)}, {@link #withDelimiter(java.lang.String)}
 * or choice of {@link #withUpperCase()} or {@link #withLowerCase()} parameters.
 * <p>
 * For primitive to hexadecimal string conversions the {@code toHexDigits}
 * methods include {@link #toHexDigits(byte)}, {@link #toHexDigits(int)}, and
 * {@link #toHexDigits(long)}, etc. The default is to use lowercase characters {@code "0-9","a-f"}.
 * For conversions producing uppercase hexadecimal the characters are {@code "0-9","A-F"}.
 * Only the {@link java.util.HexFormat#isUpperCase() HexFormat.isUpperCase()} parameter is
 * considered; the delimiter, prefix and suffix are not used.
 *
 * <p>
 * For hexadecimal string to primitive conversions the {@code fromHexDigits}
 * methods include {@link #fromHexDigits(java.lang.CharSequence) fromHexDigits(string)},
 * {@link #fromHexDigitsToLong(java.lang.CharSequence) fromHexDigitsToLong(string)}, and
 * {@link #fromHexDigit(int) fromHexDigit(int)} converts a single character or codepoint.
 * For conversions from hexadecimal characters the digits and uppercase and lowercase
 * characters in {@code "0-9", "a-f", and "A-F"} are converted to corresponding values
 * {@code 0-15}. The delimiter, prefix, suffix, and uppercase parameters are not used.
 *
 * <p>
 * For byte array to formatted hexadecimal string conversions
 * the {@code formatHex} methods include {@link #formatHex(byte[]) formatHex(byte[])}
 * and {@link #formatHex(java.lang.Appendable,byte[]) formatHex(Appendable, byte[])}.
 * The formatted output is a string or is appended to an {@link java.lang.Appendable Appendable} such as
 * {@link java.lang.StringBuilder StringBuilder} or {@link java.io.PrintStream}.
 * Each byte value is formatted as the prefix, two hexadecimal characters from the
 * uppercase or lowercase digits, and the suffix.
 * A delimiter follows each formatted value, except the last.
 * For conversions producing uppercase hexadecimal strings use {@link #withUpperCase()}.
 *
 * <p>
 * For formatted hexadecimal string to byte array conversions the
 * {@code parseHex} methods include {@link #parseHex(java.lang.CharSequence) parseHex(CharSequence)} and
 * {@link #parseHex(char[],int,int) parseHex(char[], offset, length)}.
 * Each byte value is parsed from the prefix, two case insensitive hexadecimal characters,
 * and the suffix. A delimiter follows each formatted value, except the last.
 *
 * @apiNote
 * For example, an individual byte is converted to a string of hexadecimal digits using
 * {@link java.util.HexFormat#toHexDigits(int) toHexDigits(int)} and converted from a string to a
 * primitive value using {@link java.util.HexFormat#fromHexDigits(java.lang.CharSequence) fromHexDigits(string)}.
 * <pre>{@code
 *     HexFormat hex = HexFormat.of();
 *     byte b = 127;
 *     String byteStr = hex.toHexDigits(b);
 *
 *     byte byteVal = (byte)hex.fromHexDigits(byteStr);
 *     assert(byteStr.equals("7f"));
 *     assert(b == byteVal);
 *
 *     // The hexadecimal digits are: "7f"
 * }</pre>
 * <p>
 * For a comma ({@code ", "}) separated format with a prefix ({@code "#"})
 * using lowercase hex digits the {@code HexFormat} is:
 * <pre>{@code
 *     HexFormat commaFormat = HexFormat.ofDelimiter(", ").withPrefix("#");
 *     byte[] bytes = {0, 1, 2, 3, 124, 125, 126, 127};
 *     String str = commaFormat.formatHex(bytes);
 *
 *     byte[] parsed = commaFormat.parseHex(str);
 *     assert(Arrays.equals(bytes, parsed));
 *
 *     // The formatted string is: "#00, #01, #02, #03, #7c, #7d, #7e, #7f"
 * }</pre>
 * <p>
 * For a fingerprint of byte values that uses the delimiter colon ({@code ":"})
 * and uppercase characters the {@code HexFormat} is:
 * <pre>{@code
 *     HexFormat formatFingerprint = HexFormat.ofDelimiter(":").withUpperCase();
 *     byte[] bytes = {0, 1, 2, 3, 124, 125, 126, 127};
 *     String str = formatFingerprint.formatHex(bytes);
 *     byte[] parsed = formatFingerprint.parseHex(str);
 *     assert(Arrays.equals(bytes, parsed));
 *
 *     // The formatted string is: "00:01:02:03:7C:7D:7E:7F"
 * }</pre>
 *
 * <!-- Android-removed: paragraph on ValueBased
 * <p>
 * This is a <a href="{@docRoot}/reference/java/lang/doc-files/ValueBased.html">value-based</a>
 * class; use of identity-sensitive operations (including reference equality
 * ({@code ==}), identity hash code, or synchronization) on instances of
 * {@code HexFormat} may have unpredictable results and should be avoided.
 * The {@code equals} method should be used for comparisons.
 * -->
 *
 * <p>
 * This class is immutable and thread-safe.
 * <p>
 * Unless otherwise noted, passing a null argument to any method will cause a
 * {@link java.lang.NullPointerException NullPointerException} to be thrown.
 *
 * @since 17
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class HexFormat {

HexFormat() { throw new RuntimeException("Stub!"); }

/**
 * Returns a hexadecimal formatter with no delimiter and lowercase characters.
 * The delimiter, prefix, and suffix are empty.
 * The methods {@link #withDelimiter(java.lang.String) withDelimiter},
 * {@link #withUpperCase() withUpperCase}, {@link #withLowerCase() withLowerCase},
 * {@link #withPrefix(java.lang.String) withPrefix}, and {@link #withSuffix(java.lang.String) withSuffix}
 * return copies of formatters with new parameters.
 *
 * @return a hexadecimal formatter with no delimiter and lowercase characters
 */

public static java.util.HexFormat of() { throw new RuntimeException("Stub!"); }

/**
 * Returns a hexadecimal formatter with the delimiter and lowercase characters.
 * The prefix and suffix are empty.
 * The methods {@link #withDelimiter(java.lang.String) withDelimiter},
 * {@link #withUpperCase() withUpperCase}, {@link #withLowerCase() withLowerCase},
 * {@link #withPrefix(java.lang.String) withPrefix}, and {@link #withSuffix(java.lang.String) withSuffix}
 * return copies of formatters with new parameters.
 *
 * @param delimiter a delimiter, non-null, may be empty
 * @return a {@link java.util.HexFormat HexFormat} with the delimiter and lowercase characters
 */

public static java.util.HexFormat ofDelimiter(java.lang.String delimiter) { throw new RuntimeException("Stub!"); }

/**
 * Returns a copy of this {@code HexFormat} with the delimiter.
 * @param delimiter the delimiter, non-null, may be empty
 * @return a copy of this {@code HexFormat} with the delimiter
 */

public java.util.HexFormat withDelimiter(java.lang.String delimiter) { throw new RuntimeException("Stub!"); }

/**
 * Returns a copy of this {@code HexFormat} with the prefix.
 *
 * @param prefix a prefix, non-null, may be empty
 * @return a copy of this {@code HexFormat} with the prefix
 */

public java.util.HexFormat withPrefix(java.lang.String prefix) { throw new RuntimeException("Stub!"); }

/**
 * Returns a copy of this {@code HexFormat} with the suffix.
 *
 * @param suffix a suffix, non-null, may be empty
 * @return a copy of this {@code HexFormat} with the suffix
 */

public java.util.HexFormat withSuffix(java.lang.String suffix) { throw new RuntimeException("Stub!"); }

/**
 * Returns a copy of this {@code HexFormat} to use uppercase hexadecimal characters.
 * The uppercase hexadecimal characters are {@code "0-9", "A-F"}.
 *
 * @return a copy of this {@code HexFormat} with uppercase hexadecimal characters
 */

public java.util.HexFormat withUpperCase() { throw new RuntimeException("Stub!"); }

/**
 * Returns a copy of this {@code HexFormat} to use lowercase hexadecimal characters.
 * The lowercase hexadecimal characters are {@code "0-9", "a-f"}.
 *
 * @return a copy of this {@code HexFormat} with lowercase hexadecimal characters
 */

public java.util.HexFormat withLowerCase() { throw new RuntimeException("Stub!"); }

/**
 * Returns the delimiter between hexadecimal values in formatted hexadecimal strings.
 *
 * @return the delimiter, non-null, may be empty {@code ""}
 */

public java.lang.String delimiter() { throw new RuntimeException("Stub!"); }

/**
 * Returns the prefix used for each hexadecimal value in formatted hexadecimal strings.
 *
 * @return the prefix, non-null, may be empty {@code ""}
 */

public java.lang.String prefix() { throw new RuntimeException("Stub!"); }

/**
 * Returns the suffix used for each hexadecimal value in formatted hexadecimal strings.
 *
 * @return the suffix, non-null, may be empty {@code ""}
 */

public java.lang.String suffix() { throw new RuntimeException("Stub!"); }

/**
 * Returns {@code true} if the hexadecimal digits are uppercase,
 * otherwise {@code false}.
 *
 * @return {@code true} if the hexadecimal digits are uppercase,
 *          otherwise {@code false}
 */

public boolean isUpperCase() { throw new RuntimeException("Stub!"); }

/**
 * Returns a hexadecimal string formatted from a byte array.
 * Each byte value is formatted as the prefix, two hexadecimal characters
 * {@linkplain #isUpperCase selected from} uppercase or lowercase digits, and the suffix.
 * A delimiter follows each formatted value, except the last.
 *
 * The behavior is equivalent to
 * {@link #formatHex(byte[],int,int) formatHex(bytes, 0, bytes.length))}.
 *
 * @param bytes a non-null array of bytes
 * @return a string hexadecimal formatting of the byte array
 */

public java.lang.String formatHex(byte[] bytes) { throw new RuntimeException("Stub!"); }

/**
 * Returns a hexadecimal string formatted from a byte array range.
 * Each byte value is formatted as the prefix, two hexadecimal characters
 * {@linkplain #isUpperCase selected from} uppercase or lowercase digits, and the suffix.
 * A delimiter follows each formatted value, except the last.
 *
 * @param bytes a non-null array of bytes
 * @param fromIndex the initial index of the range, inclusive
 * @param toIndex the final index of the range, exclusive
 * @return a string hexadecimal formatting each byte of the array range
 * @throws java.lang.IndexOutOfBoundsException if the array range is out of bounds
 */

public java.lang.String formatHex(byte[] bytes, int fromIndex, int toIndex) { throw new RuntimeException("Stub!"); }

/**
 * Appends formatted hexadecimal strings from a byte array to the {@link java.lang.Appendable Appendable}.
 * Each byte value is formatted as the prefix, two hexadecimal characters
 * {@linkplain #isUpperCase selected from} uppercase or lowercase digits, and the suffix.
 * A delimiter follows each formatted value, except the last.
 * The formatted hexadecimal strings are appended in zero or more calls to the {@link java.lang.Appendable Appendable} methods.
 *
 * @param <A> The type of {@code Appendable}
 * @param out an {@code Appendable}, non-null
 * @param bytes a byte array
 * @return the {@code Appendable}
 * @throws java.io.UncheckedIOException if an I/O exception occurs appending to the output
 */

public <A extends java.lang.Appendable> A formatHex(A out, byte[] bytes) { throw new RuntimeException("Stub!"); }

/**
 * Appends formatted hexadecimal strings from a byte array range to the {@link java.lang.Appendable Appendable}.
 * Each byte value is formatted as the prefix, two hexadecimal characters
 * {@linkplain #isUpperCase selected from} uppercase or lowercase digits, and the suffix.
 * A delimiter follows each formatted value, except the last.
 * The formatted hexadecimal strings are appended in zero or more calls to the {@link java.lang.Appendable Appendable} methods.
 *
 * @param <A> The type of {@code Appendable}
 * @param out an {@code Appendable}, non-null
 * @param bytes a byte array, non-null
 * @param fromIndex the initial index of the range, inclusive
 * @param toIndex the final index of the range, exclusive.
 * @return the {@code Appendable}
 * @throws java.lang.IndexOutOfBoundsException if the array range is out of bounds
 * @throws java.io.UncheckedIOException if an I/O exception occurs appending to the output
 */

public <A extends java.lang.Appendable> A formatHex(A out, byte[] bytes, int fromIndex, int toIndex) { throw new RuntimeException("Stub!"); }

/**
 * Returns a byte array containing hexadecimal values parsed from the string.
 *
 * Each byte value is parsed from the prefix, two case insensitive hexadecimal characters,
 * and the suffix. A delimiter follows each formatted value, except the last.
 * The delimiters, prefixes, and suffixes strings must be present; they may be empty strings.
 * A valid string consists only of the above format.
 *
 * @param string a string containing the byte values with prefix, hexadecimal digits, suffix,
 *            and delimiters
 * @return a byte array with the values parsed from the string
 * @throws java.lang.IllegalArgumentException if the prefix or suffix is not present for each byte value,
 *          the byte values are not hexadecimal characters, or if the delimiter is not present
 *          after all but the last byte value
 */

public byte[] parseHex(java.lang.CharSequence string) { throw new RuntimeException("Stub!"); }

/**
 * Returns a byte array containing hexadecimal values parsed from a range of the string.
 *
 * Each byte value is parsed from the prefix, two case insensitive hexadecimal characters,
 * and the suffix. A delimiter follows each formatted value, except the last.
 * The delimiters, prefixes, and suffixes strings must be present; they may be empty strings.
 * A valid string consists only of the above format.
 *
 * @param string a string range containing hexadecimal digits,
 *           delimiters, prefix, and suffix.
 * @param fromIndex the initial index of the range, inclusive
 * @param toIndex the final index of the range, exclusive.
 * @return a byte array with the values parsed from the string range
 * @throws java.lang.IllegalArgumentException if the prefix or suffix is not present for each byte value,
 *          the byte values are not hexadecimal characters, or if the delimiter is not present
 *          after all but the last byte value
 * @throws java.lang.IndexOutOfBoundsException if the string range is out of bounds
 */

public byte[] parseHex(java.lang.CharSequence string, int fromIndex, int toIndex) { throw new RuntimeException("Stub!"); }

/**
 * Returns a byte array containing hexadecimal values parsed from
 * a range of the character array.
 *
 * Each byte value is parsed from the prefix, two case insensitive hexadecimal characters,
 * and the suffix. A delimiter follows each formatted value, except the last.
 * The delimiters, prefixes, and suffixes strings must be present; they may be empty strings.
 * A valid character array range consists only of the above format.
 *
 * @param chars a character array range containing an even number of hexadecimal digits,
 *          delimiters, prefix, and suffix.
 * @param fromIndex the initial index of the range, inclusive
 * @param toIndex the final index of the range, exclusive.
 * @return a byte array with the values parsed from the character array range
 * @throws java.lang.IllegalArgumentException if the prefix or suffix is not present for each byte value,
 *          the byte values are not hexadecimal characters, or if the delimiter is not present
 *          after all but the last byte value
 * @throws java.lang.IndexOutOfBoundsException if the character array range is out of bounds
 */

public byte[] parseHex(char[] chars, int fromIndex, int toIndex) { throw new RuntimeException("Stub!"); }

/**
 * Returns the hexadecimal character for the low 4 bits of the value considering it to be a byte.
 * If the parameter {@link #isUpperCase()} is {@code true} the
 * character returned for values {@code 10-15} is uppercase {@code "A-F"},
 * otherwise the character returned is lowercase {@code "a-f"}.
 * The values in the range {@code 0-9} are returned as {@code "0-9"}.
 *
 * @param value a value, only the low 4 bits {@code 0-3} of the value are used
 * @return the hexadecimal character for the low 4 bits {@code 0-3} of the value
 */

public char toLowHexDigit(int value) { throw new RuntimeException("Stub!"); }

/**
 * Returns the hexadecimal character for the high 4 bits of the value considering it to be a byte.
 * If the parameter {@link #isUpperCase()} is {@code true} the
 * character returned for values {@code 10-15} is uppercase {@code "A-F"},
 * otherwise the character returned is lowercase {@code "a-f"}.
 * The values in the range {@code 0-9} are returned as {@code "0-9"}.
 *
 * @param value a value, only bits {@code 4-7} of the value are used
 * @return the hexadecimal character for the bits {@code 4-7} of the value
 */

public char toHighHexDigit(int value) { throw new RuntimeException("Stub!"); }

/**
 * Appends two hexadecimal characters for the byte value to the {@link java.lang.Appendable Appendable}.
 * Each nibble (4 bits) from most significant to least significant of the value
 * is formatted as if by {@link #toLowHexDigit(int) toLowHexDigit(nibble)}.
 * The hexadecimal characters are appended in one or more calls to the
 * {@link java.lang.Appendable Appendable} methods. The delimiter, prefix and suffix are not used.
 *
 * @param <A> The type of {@code Appendable}
 * @param out an {@code Appendable}, non-null
 * @param value a byte value
 * @return the {@code Appendable}
 * @throws java.io.UncheckedIOException if an I/O exception occurs appending to the output
 */

public <A extends java.lang.Appendable> A toHexDigits(A out, byte value) { throw new RuntimeException("Stub!"); }

/**
 * Returns the two hexadecimal characters for the {@code byte} value.
 * Each nibble (4 bits) from most significant to least significant of the value
 * is formatted as if by {@link #toLowHexDigit(int) toLowHexDigit(nibble)}.
 * The delimiter, prefix and suffix are not used.
 *
 * @param value a byte value
 * @return the two hexadecimal characters for the byte value
 */

public java.lang.String toHexDigits(byte value) { throw new RuntimeException("Stub!"); }

/**
 * Returns the four hexadecimal characters for the {@code char} value.
 * Each nibble (4 bits) from most significant to least significant of the value
 * is formatted as if by {@link #toLowHexDigit(int) toLowHexDigit(nibble)}.
 * The delimiter, prefix and suffix are not used.
 *
 * @param value a {@code char} value
 * @return the four hexadecimal characters for the {@code char} value
 */

public java.lang.String toHexDigits(char value) { throw new RuntimeException("Stub!"); }

/**
 * Returns the four hexadecimal characters for the {@code short} value.
 * Each nibble (4 bits) from most significant to least significant of the value
 * is formatted as if by {@link #toLowHexDigit(int) toLowHexDigit(nibble)}.
 * The delimiter, prefix and suffix are not used.
 *
 * @param value a {@code short} value
 * @return the four hexadecimal characters for the {@code short} value
 */

public java.lang.String toHexDigits(short value) { throw new RuntimeException("Stub!"); }

/**
 * Returns the eight hexadecimal characters for the {@code int} value.
 * Each nibble (4 bits) from most significant to least significant of the value
 * is formatted as if by {@link #toLowHexDigit(int) toLowHexDigit(nibble)}.
 * The delimiter, prefix and suffix are not used.
 *
 * @param value an {@code int} value
 * @return the eight hexadecimal characters for the {@code int} value
 * @see java.lang.Integer#toHexString
 */

public java.lang.String toHexDigits(int value) { throw new RuntimeException("Stub!"); }

/**
 * Returns the sixteen hexadecimal characters for the {@code long} value.
 * Each nibble (4 bits) from most significant to least significant of the value
 * is formatted as if by {@link #toLowHexDigit(int) toLowHexDigit(nibble)}.
 * The delimiter, prefix and suffix are not used.
 *
 * @param value a {@code long} value
 * @return the sixteen hexadecimal characters for the {@code long} value
 * @see java.lang.Long#toHexString
 */

public java.lang.String toHexDigits(long value) { throw new RuntimeException("Stub!"); }

/**
 * Returns up to sixteen hexadecimal characters for the {@code long} value.
 * Each nibble (4 bits) from most significant to least significant of the value
 * is formatted as if by {@link #toLowHexDigit(int) toLowHexDigit(nibble)}.
 * The delimiter, prefix and suffix are not used.
 *
 * @param value a {@code long} value
 * @param digits the number of hexadecimal digits to return, 0 to 16
 * @return the hexadecimal characters for the {@code long} value
 * @throws  java.lang.IllegalArgumentException if {@code digits} is negative or greater than 16
 */

public java.lang.String toHexDigits(long value, int digits) { throw new RuntimeException("Stub!"); }

/**
 * Returns {@code true} if the character is a valid hexadecimal character or codepoint.
 * The valid hexadecimal characters are:
 * <ul>
 * <li>{@code '0' ('\u005Cu0030')} through {@code '9' ('\u005Cu0039')} inclusive,
 * <li>{@code 'A' ('\u005Cu0041')} through {@code 'F' ('\u005Cu0046')} inclusive, and
 * <li>{@code 'a' ('\u005Cu0061')} through {@code 'f' ('\u005Cu0066')} inclusive.
 * </ul>
 * @param ch a codepoint
 * @return {@code true} if the character is valid a hexadecimal character,
 *          otherwise {@code false}
 */

public static boolean isHexDigit(int ch) { throw new RuntimeException("Stub!"); }

/**
 * Returns the value for the hexadecimal character or codepoint.
 * The value is:
 * <ul>
 * <li>{@code (ch - '0')} for {@code '0'} through {@code '9'} inclusive,
 * <li>{@code (ch - 'A' + 10)} for {@code 'A'} through {@code 'F'} inclusive, and
 * <li>{@code (ch - 'a' + 10)} for {@code 'a'} through {@code 'f'} inclusive.
 * </ul>
 *
 * @param ch a character or codepoint
 * @return the value {@code 0-15}
 * @throws  java.lang.NumberFormatException if the codepoint is not a hexadecimal character
 */

public static int fromHexDigit(int ch) { throw new RuntimeException("Stub!"); }

/**
 * Returns the {@code int} value parsed from a string of up to eight hexadecimal characters.
 * The hexadecimal characters are parsed from most significant to least significant
 * using {@link #fromHexDigit(int)} to form an unsigned value.
 * The value is zero extended to 32 bits and is returned as an {@code int}.
 *
 * @apiNote
 * {@link java.lang.Integer#parseInt(java.lang.String,int) Integer.parseInt(s, 16)} and
 * {@link java.lang.Integer#parseUnsignedInt(java.lang.String,int) Integer.parseUnsignedInt(s, 16)}
 * are similar but allow all Unicode hexadecimal digits defined by
 * {@link java.lang.Character#digit(char,int) Character.digit(ch, 16)}.
 * {@code HexFormat} uses only hexadecimal characters
 * {@code "0-9"}, {@code "A-F"} and {@code "a-f"}.
 * Signed hexadecimal strings can be parsed with {@link java.lang.Integer#parseInt(java.lang.String,int) Integer#parseInt(String, int)}.
 *
 * @param string a CharSequence containing up to eight hexadecimal characters
 * @return the value parsed from the string
 * @throws  java.lang.IllegalArgumentException if the string length is greater than eight (8) or
 *      if any of the characters is not a hexadecimal character
 */

public static int fromHexDigits(java.lang.CharSequence string) { throw new RuntimeException("Stub!"); }

/**
 * Returns the {@code int} value parsed from a string range of up to eight hexadecimal
 * characters.
 * The characters in the range {@code fromIndex} to {@code toIndex}, exclusive,
 * are parsed from most significant to least significant
 * using {@link #fromHexDigit(int)} to form an unsigned value.
 * The value is zero extended to 32 bits and is returned as an {@code int}.
 *
 * @apiNote
 * {@link java.lang.Integer#parseInt(java.lang.String,int) Integer.parseInt(s, 16)} and
 * {@link java.lang.Integer#parseUnsignedInt(java.lang.String,int) Integer.parseUnsignedInt(s, 16)}
 * are similar but allow all Unicode hexadecimal digits defined by
 * {@link java.lang.Character#digit(char,int) Character.digit(ch, 16)}.
 * {@code HexFormat} uses only hexadecimal characters
 * {@code "0-9"}, {@code "A-F"} and {@code "a-f"}.
 * Signed hexadecimal strings can be parsed with {@link java.lang.Integer#parseInt(java.lang.String,int) Integer#parseInt(String, int)}.
 *
 * @param string a CharSequence containing the characters
 * @param fromIndex the initial index of the range, inclusive
 * @param toIndex the final index of the range, exclusive.
 * @return the value parsed from the string range
 * @throws  java.lang.IndexOutOfBoundsException if the range is out of bounds
 *          for the {@code CharSequence}
 * @throws  java.lang.IllegalArgumentException if length of the range is greater than eight (8) or
 *          if any of the characters is not a hexadecimal character
 */

public static int fromHexDigits(java.lang.CharSequence string, int fromIndex, int toIndex) { throw new RuntimeException("Stub!"); }

/**
 * Returns the long value parsed from a string of up to sixteen hexadecimal characters.
 * The hexadecimal characters are parsed from most significant to least significant
 * using {@link #fromHexDigit(int)} to form an unsigned value.
 * The value is zero extended to 64 bits and is returned as a {@code long}.
 *
 * @apiNote
 * {@link java.lang.Long#parseLong(java.lang.String,int) Long.parseLong(s, 16)} and
 * {@link java.lang.Long#parseUnsignedLong(java.lang.String,int) Long.parseUnsignedLong(s, 16)}
 * are similar but allow all Unicode hexadecimal digits defined by
 * {@link java.lang.Character#digit(char,int) Character.digit(ch, 16)}.
 * {@code HexFormat} uses only hexadecimal characters
 * {@code "0-9"}, {@code "A-F"} and {@code "a-f"}.
 * Signed hexadecimal strings can be parsed with {@link java.lang.Long#parseLong(java.lang.String,int) Long#parseLong(String, int)}.
 *
 * @param string a CharSequence containing up to sixteen hexadecimal characters
 * @return the value parsed from the string
 * @throws  java.lang.IllegalArgumentException if the string length is greater than sixteen (16) or
 *         if any of the characters is not a hexadecimal character
 */

public static long fromHexDigitsToLong(java.lang.CharSequence string) { throw new RuntimeException("Stub!"); }

/**
 * Returns the long value parsed from a string range of up to sixteen hexadecimal
 * characters.
 * The characters in the range {@code fromIndex} to {@code toIndex}, exclusive,
 * are parsed from most significant to least significant
 * using {@link #fromHexDigit(int)} to form an unsigned value.
 * The value is zero extended to 64 bits and is returned as a {@code long}.
 *
 * @apiNote
 * {@link java.lang.Long#parseLong(java.lang.String,int) Long.parseLong(s, 16)} and
 * {@link java.lang.Long#parseUnsignedLong(java.lang.String,int) Long.parseUnsignedLong(s, 16)}
 * are similar but allow all Unicode hexadecimal digits defined by
 * {@link java.lang.Character#digit(char,int) Character.digit(ch, 16)}.
 * {@code HexFormat} uses only hexadecimal characters
 * {@code "0-9"}, {@code "A-F"} and {@code "a-f"}.
 * Signed hexadecimal strings can be parsed with {@link java.lang.Long#parseLong(java.lang.String,int) Long#parseLong(String, int)}.
 *
 * @param string a CharSequence containing the characters
 * @param fromIndex the initial index of the range, inclusive
 * @param toIndex the final index of the range, exclusive.
 * @return the value parsed from the string range
 * @throws  java.lang.IndexOutOfBoundsException if the range is out of bounds
 *          for the {@code CharSequence}
 * @throws  java.lang.IllegalArgumentException if the length of the range is greater than sixteen (16) or
 *          if any of the characters is not a hexadecimal character
 */

public static long fromHexDigitsToLong(java.lang.CharSequence string, int fromIndex, int toIndex) { throw new RuntimeException("Stub!"); }

/**
 * Returns {@code true} if the other object is a {@code HexFormat}
 * with the same parameters.
 *
 * @param o an object, may be null
 * @return {@code true} if the other object is a {@code HexFormat} and the parameters
 *         uppercase, delimiter, prefix, and suffix are equal;
 *         otherwise {@code false}
 */

public boolean equals(java.lang.Object o) { throw new RuntimeException("Stub!"); }

/**
 * Returns a hashcode for this {@code HexFormat}.
 *
 * @return a hashcode for this {@code HexFormat}
 */

public int hashCode() { throw new RuntimeException("Stub!"); }

/**
 * Returns a description of the formatter parameters for uppercase,
 * delimiter, prefix, and suffix.
 *
 * @return a description of this {@code HexFormat}
 */

public java.lang.String toString() { throw new RuntimeException("Stub!"); }
}

