/*
 * Copyright (c) 2012, 2020, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package java.util;

import java.util.function.DoubleConsumer;
import java.util.stream.DoubleStream;

/**
 * A container object which may or may not contain a {@code double} value.
 * If a value is present, {@code isPresent()} returns {@code true}. If no
 * value is present, the object is considered <i>empty</i> and
 * {@code isPresent()} returns {@code false}.
 *
 * <p>Additional methods that depend on the presence or absence of a contained
 * value are provided, such as {@link #orElse(double) orElse()}
 * (returns a default value if no value is present) and
 * {@link #ifPresent(java.util.function.DoubleConsumer) ifPresent()} (performs
 * an action if a value is present).
 *
 * <p>This is a <a href="https://docs.oracle.com/en/java/javase/17/docs/api/java.base/java/lang/doc-files/ValueBased.html">value-based</a>
 * class; programmers should treat instances that are
 * {@linkplain #equals(java.lang.Object) equal} as interchangeable and should not
 * use instances for synchronization, or unpredictable behavior may
 * occur. For example, in a future release, synchronization may fail.
 *
 * @apiNote
 * {@code OptionalDouble} is primarily intended for use as a method return type where
 * there is a clear need to represent "no result." A variable whose type is
 * {@code OptionalDouble} should never itself be {@code null}; it should always point
 * to an {@code OptionalDouble} instance.
 *
 * @since 1.8
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class OptionalDouble {

OptionalDouble() { throw new RuntimeException("Stub!"); }

/**
 * Returns an empty {@code OptionalDouble} instance.  No value is present
 * for this {@code OptionalDouble}.
 *
 * @apiNote
 * Though it may be tempting to do so, avoid testing if an object is empty
 * by comparing with {@code ==} or {@code !=} against instances returned by
 * {@code OptionalDouble.empty()}.  There is no guarantee that it is a singleton.
 * Instead, use {@link #isEmpty()} or {@link #isPresent()}.
 *
 *  @return an empty {@code OptionalDouble}.
 */

public static java.util.OptionalDouble empty() { throw new RuntimeException("Stub!"); }

/**
 * Returns an {@code OptionalDouble} describing the given value.
 *
 * @param value the value to describe
 * @return an {@code OptionalDouble} with the value present
 */

public static java.util.OptionalDouble of(double value) { throw new RuntimeException("Stub!"); }

/**
 * If a value is present, returns the value, otherwise throws
 * {@code NoSuchElementException}.
 *
 * @apiNote
 * The preferred alternative to this method is {@link #orElseThrow()}.
 *
 * @return the value described by this {@code OptionalDouble}
 * @throws java.util.NoSuchElementException if no value is present
 */

public double getAsDouble() { throw new RuntimeException("Stub!"); }

/**
 * If a value is present, returns {@code true}, otherwise {@code false}.
 *
 * @return {@code true} if a value is present, otherwise {@code false}
 */

public boolean isPresent() { throw new RuntimeException("Stub!"); }

/**
 * If a value is not present, returns {@code true}, otherwise
 * {@code false}.
 *
 * @return  {@code true} if a value is not present, otherwise {@code false}
 * @since   11
 */

public boolean isEmpty() { throw new RuntimeException("Stub!"); }

/**
 * If a value is present, performs the given action with the value,
 * otherwise does nothing.
 *
 * @param action the action to be performed, if a value is present
 * @throws java.lang.NullPointerException if value is present and the given action is
 *         {@code null}
 */

public void ifPresent(java.util.function.DoubleConsumer action) { throw new RuntimeException("Stub!"); }

/**
 * If a value is present, performs the given action with the value,
 * otherwise performs the given empty-based action.
 *
 * @param action the action to be performed, if a value is present
 * @param emptyAction the empty-based action to be performed, if no value is
 * present
 * @throws java.lang.NullPointerException if a value is present and the given action
 *         is {@code null}, or no value is present and the given empty-based
 *         action is {@code null}.
 * @since 9
 */

public void ifPresentOrElse(java.util.function.DoubleConsumer action, java.lang.Runnable emptyAction) { throw new RuntimeException("Stub!"); }

/**
 * If a value is present, returns a sequential {@link java.util.stream.DoubleStream DoubleStream}
 * containing only that value, otherwise returns an empty
 * {@code DoubleStream}.
 *
 * @apiNote
 * This method can be used to transform a {@code Stream} of optional doubles
 * to a {@code DoubleStream} of present doubles:
 * <pre>{@code
 *     Stream<OptionalDouble> os = ..
 *     DoubleStream s = os.flatMapToDouble(OptionalDouble::stream)
 * }</pre>
 *
 * @return the optional value as a {@code DoubleStream}
 * @since 9
 */

public java.util.stream.DoubleStream stream() { throw new RuntimeException("Stub!"); }

/**
 * If a value is present, returns the value, otherwise returns
 * {@code other}.
 *
 * @param other the value to be returned, if no value is present
 * @return the value, if present, otherwise {@code other}
 */

public double orElse(double other) { throw new RuntimeException("Stub!"); }

/**
 * If a value is present, returns the value, otherwise returns the result
 * produced by the supplying function.
 *
 * @param supplier the supplying function that produces a value to be returned
 * @return the value, if present, otherwise the result produced by the
 *         supplying function
 * @throws java.lang.NullPointerException if no value is present and the supplying
 *         function is {@code null}
 */

public double orElseGet(java.util.function.DoubleSupplier supplier) { throw new RuntimeException("Stub!"); }

/**
 * If a value is present, returns the value, otherwise throws
 * {@code NoSuchElementException}.
 *
 * @return the value described by this {@code OptionalDouble}
 * @throws java.util.NoSuchElementException if no value is present
 * @since 10
 */

public double orElseThrow() { throw new RuntimeException("Stub!"); }

/**
 * If a value is present, returns the value, otherwise throws an exception
 * produced by the exception supplying function.
 *
 * @apiNote
 * A method reference to the exception constructor with an empty argument
 * list can be used as the supplier. For example,
 * {@code IllegalStateException::new}
 *
 * @param <X> Type of the exception to be thrown
 * @param exceptionSupplier the supplying function that produces an
 *        exception to be thrown
 * @return the value, if present
 * @throws X if no value is present
 * @throws java.lang.NullPointerException if no value is present and the exception
 *         supplying function is {@code null}
 */

public <X extends java.lang.Throwable> double orElseThrow(java.util.function.Supplier<? extends X> exceptionSupplier) throws X { throw new RuntimeException("Stub!"); }

/**
 * Indicates whether some other object is "equal to" this
 * {@code OptionalDouble}. The other object is considered equal if:
 * <ul>
 * <li>it is also an {@code OptionalDouble} and;
 * <li>both instances have no value present or;
 * <li>the present values are "equal to" each other via
 * {@code Double.compare() == 0}.
 * </ul>
 *
 * @param obj an object to be tested for equality
 * @return {@code true} if the other object is "equal to" this object
 *         otherwise {@code false}
 */

public boolean equals(java.lang.Object obj) { throw new RuntimeException("Stub!"); }

/**
 * Returns the hash code of the value, if present, otherwise {@code 0}
 * (zero) if no value is present.
 *
 * @return hash code value of the present value or {@code 0} if no value is
 *         present
 */

public int hashCode() { throw new RuntimeException("Stub!"); }

/**
 * Returns a non-empty string representation of this {@code OptionalDouble}
 * suitable for debugging.  The exact presentation format is unspecified and
 * may vary between implementations and versions.
 *
 * @implSpec
 * If a value is present the result must include its string representation
 * in the result.  Empty and present {@code OptionalDouble}s must be
 * unambiguously differentiable.
 *
 * @return the string representation of this instance
 */

public java.lang.String toString() { throw new RuntimeException("Stub!"); }
}

