/*
 * Copyright (C) 2014 The Android Open Source Project
 * Copyright (c) 2003, 2020, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package java.util;

import java.io.*;
import java.math.*;
import java.nio.*;
import java.nio.channels.*;
import java.nio.charset.*;
import java.nio.file.Path;
import java.text.*;
import java.util.regex.*;

/**
 * A simple text scanner which can parse primitive types and strings using
 * regular expressions.
 *
 * <p>A {@code Scanner} breaks its input into tokens using a
 * delimiter pattern, which by default matches whitespace. The resulting
 * tokens may then be converted into values of different types using the
 * various {@code next} methods.
 *
 * <p>For example, this code allows a user to read a number from
 * {@code System.in}:
 * <blockquote><pre>{@code
 *     Scanner sc = new Scanner(System.in);
 *     int i = sc.nextInt();
 * }</pre></blockquote>
 *
 * <p>As another example, this code allows {@code long} types to be
 * assigned from entries in a file {@code myNumbers}:
 * <blockquote><pre>{@code
 *      Scanner sc = new Scanner(new File("myNumbers"));
 *      while (sc.hasNextLong()) {
 *          long aLong = sc.nextLong();
 *      }
 * }</pre></blockquote>
 *
 * <p>The scanner can also use delimiters other than whitespace. This
 * example reads several items in from a string:
 * <blockquote><pre>{@code
 *     String input = "1 fish 2 fish red fish blue fish";
 *     Scanner s = new Scanner(input).useDelimiter("\\s*fish\\s*");
 *     System.out.println(s.nextInt());
 *     System.out.println(s.nextInt());
 *     System.out.println(s.next());
 *     System.out.println(s.next());
 *     s.close();
 * }</pre></blockquote>
 * <p>
 * prints the following output:
 * <blockquote><pre>{@code
 *     1
 *     2
 *     red
 *     blue
 * }</pre></blockquote>
 *
 * <p>The same output can be generated with this code, which uses a regular
 * expression to parse all four tokens at once:
 * <blockquote><pre>{@code
 *     String input = "1 fish 2 fish red fish blue fish";
 *     Scanner s = new Scanner(input);
 *     s.findInLine("(\\d+) fish (\\d+) fish (\\w+) fish (\\w+)");
 *     MatchResult result = s.match();
 *     for (int i=1; i<=result.groupCount(); i++)
 *         System.out.println(result.group(i));
 *     s.close();
 * }</pre></blockquote>
 *
 * <p>The <a id="default-delimiter">default whitespace delimiter</a> used
 * by a scanner is as recognized by {@link java.lang.Character#isWhitespace(char)  Character.isWhitespace()}. The {@link #reset reset()}
 * method will reset the value of the scanner's delimiter to the default
 * whitespace delimiter regardless of whether it was previously changed.
 *
 * <p>A scanning operation may block waiting for input.
 *
 * <p>The {@link #next} and {@link #hasNext} methods and their
 * companion methods (such as {@link #nextInt} and
 * {@link #hasNextInt}) first skip any input that matches the delimiter
 * pattern, and then attempt to return the next token. Both {@code hasNext()}
 * and {@code next()} methods may block waiting for further input.  Whether a
 * {@code hasNext()} method blocks has no connection to whether or not its
 * associated {@code next()} method will block. The {@link #tokens} method
 * may also block waiting for input.
 *
 * <p>The {@link #findInLine findInLine()},
 * {@link #findWithinHorizon findWithinHorizon()},
 * {@link #skip skip()}, and {@link #findAll findAll()}
 * methods operate independently of the delimiter pattern. These methods will
 * attempt to match the specified pattern with no regard to delimiters in the
 * input and thus can be used in special circumstances where delimiters are
 * not relevant. These methods may block waiting for more input.
 *
 * <p>When a scanner throws an {@link java.util.InputMismatchException InputMismatchException}, the scanner
 * will not pass the token that caused the exception, so that it may be
 * retrieved or skipped via some other method.
 *
 * <p>Depending upon the type of delimiting pattern, empty tokens may be
 * returned. For example, the pattern {@code "\\s+"} will return no empty
 * tokens since it matches multiple instances of the delimiter. The delimiting
 * pattern {@code "\\s"} could return empty tokens since it only passes one
 * space at a time.
 *
 * <p> A scanner can read text from any object which implements the {@link
 * java.lang.Readable} interface.  If an invocation of the underlying
 * readable's {@link java.lang.Readable#read read()} method throws an {@link
 * java.io.IOException} then the scanner assumes that the end of the input
 * has been reached.  The most recent {@code IOException} thrown by the
 * underlying readable can be retrieved via the {@link #ioException} method.
 *
 * <p>When a {@code Scanner} is closed, it will close its input source
 * if the source implements the {@link java.io.Closeable} interface.
 *
 * <p>A {@code Scanner} is not safe for multithreaded use without
 * external synchronization.
 *
 * <p>Unless otherwise mentioned, passing a {@code null} parameter into
 * any method of a {@code Scanner} will cause a
 * {@code NullPointerException} to be thrown.
 *
 * <p>A scanner will default to interpreting numbers as decimal unless a
 * different radix has been set by using the {@link #useRadix} method. The
 * {@link #reset} method will reset the value of the scanner's radix to
 * {@code 10} regardless of whether it was previously changed.
 *
 * <h2> <a id="localized-numbers">Localized numbers</a> </h2>
 *
 * <p> An instance of this class is capable of scanning numbers in the standard
 * formats as well as in the formats of the scanner's locale. A scanner's
 * <a id="initial-locale">initial locale </a>is the value returned by the {@link
 * java.util.Locale#getDefault(Locale.Category)
 * Locale.getDefault(Locale.Category.FORMAT)} method; it may be changed via the {@link
 * #useLocale useLocale()} method. The {@link #reset} method will reset the value of the
 * scanner's locale to the initial locale regardless of whether it was
 * previously changed.
 *
 * <p>The localized formats are defined in terms of the following parameters,
 * which for a particular locale are taken from that locale's {@link
 * java.text.DecimalFormat DecimalFormat} object, {@code df}, and its and
 * {@link java.text.DecimalFormatSymbols DecimalFormatSymbols} object,
 * {@code dfs}.
 *
 * <blockquote><dl>
 *     <dt><i>LocalGroupSeparator&nbsp;&nbsp;</i>
 *         <dd>The character used to separate thousands groups,
 *         <i>i.e.,</i>&nbsp;{@code dfs.}{@link
 *         java.text.DecimalFormatSymbols#getGroupingSeparator
 *         getGroupingSeparator()}
 *     <dt><i>LocalDecimalSeparator&nbsp;&nbsp;</i>
 *         <dd>The character used for the decimal point,
 *     <i>i.e.,</i>&nbsp;{@code dfs.}{@link
 *     java.text.DecimalFormatSymbols#getDecimalSeparator
 *     getDecimalSeparator()}
 *     <dt><i>LocalPositivePrefix&nbsp;&nbsp;</i>
 *         <dd>The string that appears before a positive number (may
 *         be empty), <i>i.e.,</i>&nbsp;{@code df.}{@link
 *         java.text.DecimalFormat#getPositivePrefix
 *         getPositivePrefix()}
 *     <dt><i>LocalPositiveSuffix&nbsp;&nbsp;</i>
 *         <dd>The string that appears after a positive number (may be
 *         empty), <i>i.e.,</i>&nbsp;{@code df.}{@link
 *         java.text.DecimalFormat#getPositiveSuffix
 *         getPositiveSuffix()}
 *     <dt><i>LocalNegativePrefix&nbsp;&nbsp;</i>
 *         <dd>The string that appears before a negative number (may
 *         be empty), <i>i.e.,</i>&nbsp;{@code df.}{@link
 *         java.text.DecimalFormat#getNegativePrefix
 *         getNegativePrefix()}
 *     <dt><i>LocalNegativeSuffix&nbsp;&nbsp;</i>
 *         <dd>The string that appears after a negative number (may be
 *         empty), <i>i.e.,</i>&nbsp;{@code df.}{@link
 *     java.text.DecimalFormat#getNegativeSuffix
 *     getNegativeSuffix()}
 *     <dt><i>LocalNaN&nbsp;&nbsp;</i>
 *         <dd>The string that represents not-a-number for
 *         floating-point values,
 *         <i>i.e.,</i>&nbsp;{@code dfs.}{@link
 *         java.text.DecimalFormatSymbols#getNaN
 *         getNaN()}
 *     <dt><i>LocalInfinity&nbsp;&nbsp;</i>
 *         <dd>The string that represents infinity for floating-point
 *         values, <i>i.e.,</i>&nbsp;{@code dfs.}{@link
 *         java.text.DecimalFormatSymbols#getInfinity
 *         getInfinity()}
 * </dl></blockquote>
 *
 * <h3> <a id="number-syntax">Number syntax</a> </h3>
 *
 * <p> The strings that can be parsed as numbers by an instance of this class
 * are specified in terms of the following regular-expression grammar, where
 * Rmax is the highest digit in the radix being used (for example, Rmax is 9 in base 10).
 *
 * <dl>
 *   <dt><i>NonAsciiDigit</i>:
 *       <dd>A non-ASCII character c for which
 *            {@link java.lang.Character#isDigit Character.isDigit}{@code (c)}
 *                        returns&nbsp;true
 *
 *   <dt><i>Non0Digit</i>:
 *       <dd>{@code [1-}<i>Rmax</i>{@code ] | }<i>NonASCIIDigit</i>
 *
 *   <dt><i>Digit</i>:
 *       <dd>{@code [0-}<i>Rmax</i>{@code ] | }<i>NonASCIIDigit</i>
 *
 *   <dt><i>GroupedNumeral</i>:
 *       <dd><code>(&nbsp;</code><i>Non0Digit</i>
 *                   <i>Digit</i>{@code ?
 *                   }<i>Digit</i>{@code ?}
 *       <dd>&nbsp;&nbsp;&nbsp;&nbsp;<code>(&nbsp;</code><i>LocalGroupSeparator</i>
 *                         <i>Digit</i>
 *                         <i>Digit</i>
 *                         <i>Digit</i>{@code  )+ )}
 *
 *   <dt><i>Numeral</i>:
 *       <dd>{@code ( ( }<i>Digit</i>{@code + )
 *               | }<i>GroupedNumeral</i>{@code  )}
 *
 *   <dt><a id="Integer-regex"><i>Integer</i>:</a>
 *       <dd>{@code ( [-+]? ( }<i>Numeral</i>{@code
 *                               ) )}
 *       <dd>{@code | }<i>LocalPositivePrefix</i> <i>Numeral</i>
 *                      <i>LocalPositiveSuffix</i>
 *       <dd>{@code | }<i>LocalNegativePrefix</i> <i>Numeral</i>
 *                 <i>LocalNegativeSuffix</i>
 *
 *   <dt><i>DecimalNumeral</i>:
 *       <dd><i>Numeral</i>
 *       <dd>{@code | }<i>Numeral</i>
 *                 <i>LocalDecimalSeparator</i>
 *                 <i>Digit</i>{@code *}
 *       <dd>{@code | }<i>LocalDecimalSeparator</i>
 *                 <i>Digit</i>{@code +}
 *
 *   <dt><i>Exponent</i>:
 *       <dd>{@code ( [eE] [+-]? }<i>Digit</i>{@code + )}
 *
 *   <dt><a id="Decimal-regex"><i>Decimal</i>:</a>
 *       <dd>{@code ( [-+]? }<i>DecimalNumeral</i>
 *                         <i>Exponent</i>{@code ? )}
 *       <dd>{@code | }<i>LocalPositivePrefix</i>
 *                 <i>DecimalNumeral</i>
 *                 <i>LocalPositiveSuffix</i>
 *                 <i>Exponent</i>{@code ?}
 *       <dd>{@code | }<i>LocalNegativePrefix</i>
 *                 <i>DecimalNumeral</i>
 *                 <i>LocalNegativeSuffix</i>
 *                 <i>Exponent</i>{@code ?}
 *
 *   <dt><i>HexFloat</i>:
 *       <dd>{@code [-+]? 0[xX][0-9a-fA-F]*\.[0-9a-fA-F]+
 *                 ([pP][-+]?[0-9]+)?}
 *
 *   <dt><i>NonNumber</i>:
 *       <dd>{@code NaN
 *                          | }<i>LocalNan</i>{@code
 *                          | Infinity
 *                          | }<i>LocalInfinity</i>
 *
 *   <dt><i>SignedNonNumber</i>:
 *       <dd>{@code ( [-+]? }<i>NonNumber</i>{@code  )}
 *       <dd>{@code | }<i>LocalPositivePrefix</i>
 *                 <i>NonNumber</i>
 *                 <i>LocalPositiveSuffix</i>
 *       <dd>{@code | }<i>LocalNegativePrefix</i>
 *                 <i>NonNumber</i>
 *                 <i>LocalNegativeSuffix</i>
 *
 *   <dt><a id="Float-regex"><i>Float</i></a>:
 *       <dd><i>Decimal</i>
 *           {@code | }<i>HexFloat</i>
 *           {@code | }<i>SignedNonNumber</i>
 *
 * </dl>
 * <p>Whitespace is not significant in the above regular expressions.
 *
 * @since   1.5
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class Scanner implements java.util.Iterator<java.lang.String>, java.io.Closeable {

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified source.
 *
 * @param  source A character source implementing the {@link java.lang.Readable Readable}
 *         interface
 */

public Scanner(java.lang.Readable source) { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified input stream. Bytes from the stream are converted
 * into characters using the underlying platform's
 * {@linkplain java.nio.charset.Charset#defaultCharset() default charset}.
 *
 * @param  source An input stream to be scanned
 */

public Scanner(java.io.InputStream source) { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified input stream. Bytes from the stream are converted
 * into characters using the specified charset.
 *
 * @param  source An input stream to be scanned
 * @param charsetName The encoding type used to convert bytes from the
 *        stream into characters to be scanned
 * @throws java.lang.IllegalArgumentException if the specified character set
 *         does not exist
 */

public Scanner(java.io.InputStream source, java.lang.String charsetName) { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified input stream. Bytes from the stream are converted
 * into characters using the specified charset.
 *
 * @param  source an input stream to be scanned
 * @param  charset the charset used to convert bytes from the file
 *         into characters to be scanned
 * @since  10
 */

public Scanner(java.io.InputStream source, java.nio.charset.Charset charset) { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified file. Bytes from the file are converted into
 * characters using the underlying platform's
 * {@linkplain java.nio.charset.Charset#defaultCharset() default charset}.
 *
 * @param  source A file to be scanned
 * @throws java.io.FileNotFoundException if source is not found
 */

public Scanner(java.io.File source) throws java.io.FileNotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified file. Bytes from the file are converted into
 * characters using the specified charset.
 *
 * @param  source A file to be scanned
 * @param charsetName The encoding type used to convert bytes from the file
 *        into characters to be scanned
 * @throws java.io.FileNotFoundException if source is not found
 * @throws java.lang.IllegalArgumentException if the specified encoding is
 *         not found
 */

public Scanner(java.io.File source, java.lang.String charsetName) throws java.io.FileNotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified file. Bytes from the file are converted into
 * characters using the specified charset.
 *
 * @param  source A file to be scanned
 * @param  charset The charset used to convert bytes from the file
 *         into characters to be scanned
 * @throws java.io.IOException
 *         if an I/O error occurs opening the source
 * @since  10
 */

public Scanner(java.io.File source, java.nio.charset.Charset charset) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified file. Bytes from the file are converted into
 * characters using the underlying platform's
 * {@linkplain java.nio.charset.Charset#defaultCharset() default charset}.
 *
 * @param   source
 *          the path to the file to be scanned
 * @throws  java.io.IOException
 *          if an I/O error occurs opening source
 *
 * @since   1.7
 */

public Scanner(java.nio.file.Path source) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified file. Bytes from the file are converted into
 * characters using the specified charset.
 *
 * @param   source
 *          the path to the file to be scanned
 * @param   charsetName
 *          The encoding type used to convert bytes from the file
 *          into characters to be scanned
 * @throws  java.io.IOException
 *          if an I/O error occurs opening source
 * @throws  java.lang.IllegalArgumentException
 *          if the specified encoding is not found
 * @since   1.7
 */

public Scanner(java.nio.file.Path source, java.lang.String charsetName) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified file. Bytes from the file are converted into
 * characters using the specified charset.
 *
 * @param   source
 *          the path to the file to be scanned
 * @param   charset
 *          the charset used to convert bytes from the file
 *          into characters to be scanned
 * @throws  java.io.IOException
 *          if an I/O error occurs opening the source
 * @since   10
 */

public Scanner(java.nio.file.Path source, java.nio.charset.Charset charset) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified string.
 *
 * @param  source A string to scan
 */

public Scanner(java.lang.String source) { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified channel. Bytes from the source are converted into
 * characters using the underlying platform's
 * {@linkplain java.nio.charset.Charset#defaultCharset() default charset}.
 *
 * @param  source A channel to scan
 */

public Scanner(java.nio.channels.ReadableByteChannel source) { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified channel. Bytes from the source are converted into
 * characters using the specified charset.
 *
 * @param  source A channel to scan
 * @param charsetName The encoding type used to convert bytes from the
 *        channel into characters to be scanned
 * @throws java.lang.IllegalArgumentException if the specified character set
 *         does not exist
 */

public Scanner(java.nio.channels.ReadableByteChannel source, java.lang.String charsetName) { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@code Scanner} that produces values scanned
 * from the specified channel. Bytes from the source are converted into
 * characters using the specified charset.
 *
 * @param source a channel to scan
 * @param charset the encoding type used to convert bytes from the
 *        channel into characters to be scanned
 * @since 10
 */

public Scanner(java.nio.channels.ReadableByteChannel source, java.nio.charset.Charset charset) { throw new RuntimeException("Stub!"); }

/**
 * Closes this scanner.
 *
 * <p> If this scanner has not yet been closed then if its underlying
 * {@linkplain java.lang.Readable readable} also implements the {@link
 * java.io.Closeable} interface then the readable's {@code close} method
 * will be invoked.  If this scanner is already closed then invoking this
 * method will have no effect.
 *
 * <p>Attempting to perform search operations after a scanner has
 * been closed will result in an {@link java.lang.IllegalStateException IllegalStateException}.
 *
 */

public void close() { throw new RuntimeException("Stub!"); }

/**
 * Returns the {@code IOException} last thrown by this
 * {@code Scanner}'s underlying {@code Readable}. This method
 * returns {@code null} if no such exception exists.
 *
 * @return the last exception thrown by this scanner's readable
 */

public java.io.IOException ioException() { throw new RuntimeException("Stub!"); }

/**
 * Returns the {@code Pattern} this {@code Scanner} is currently
 * using to match delimiters.
 *
 * @return this scanner's delimiting pattern.
 */

public java.util.regex.Pattern delimiter() { throw new RuntimeException("Stub!"); }

/**
 * Sets this scanner's delimiting pattern to the specified pattern.
 *
 * @param pattern A delimiting pattern
 * @return this scanner
 */

public java.util.Scanner useDelimiter(java.util.regex.Pattern pattern) { throw new RuntimeException("Stub!"); }

/**
 * Sets this scanner's delimiting pattern to a pattern constructed from
 * the specified {@code String}.
 *
 * <p> An invocation of this method of the form
 * {@code useDelimiter(pattern)} behaves in exactly the same way as the
 * invocation {@code useDelimiter(Pattern.compile(pattern))}.
 *
 * <p> Invoking the {@link #reset} method will set the scanner's delimiter
 * to the <a href= "#default-delimiter">default</a>.
 *
 * @param pattern A string specifying a delimiting pattern
 * @return this scanner
 */

public java.util.Scanner useDelimiter(java.lang.String pattern) { throw new RuntimeException("Stub!"); }

/**
 * Returns this scanner's locale.
 *
 * <p>A scanner's locale affects many elements of its default
 * primitive matching regular expressions; see
 * <a href= "#localized-numbers">localized numbers</a> above.
 *
 * @return this scanner's locale
 */

public java.util.Locale locale() { throw new RuntimeException("Stub!"); }

/**
 * Sets this scanner's locale to the specified locale.
 *
 * <p>A scanner's locale affects many elements of its default
 * primitive matching regular expressions; see
 * <a href= "#localized-numbers">localized numbers</a> above.
 *
 * <p>Invoking the {@link #reset} method will set the scanner's locale to
 * the <a href= "#initial-locale">initial locale</a>.
 *
 * @param locale A string specifying the locale to use
 * @return this scanner
 */

public java.util.Scanner useLocale(java.util.Locale locale) { throw new RuntimeException("Stub!"); }

/**
 * Returns this scanner's default radix.
 *
 * <p>A scanner's radix affects elements of its default
 * number matching regular expressions; see
 * <a href= "#localized-numbers">localized numbers</a> above.
 *
 * @return the default radix of this scanner
 */

public int radix() { throw new RuntimeException("Stub!"); }

/**
 * Sets this scanner's default radix to the specified radix.
 *
 * <p>A scanner's radix affects elements of its default
 * number matching regular expressions; see
 * <a href= "#localized-numbers">localized numbers</a> above.
 *
 * <p>If the radix is less than {@link java.lang.Character#MIN_RADIX Character.MIN_RADIX}
 * or greater than {@link java.lang.Character#MAX_RADIX Character.MAX_RADIX}, then an
 * {@code IllegalArgumentException} is thrown.
 *
 * <p>Invoking the {@link #reset} method will set the scanner's radix to
 * {@code 10}.
 *
 * @param radix The radix to use when scanning numbers
 * @return this scanner
 * @throws java.lang.IllegalArgumentException if radix is out of range
 */

public java.util.Scanner useRadix(int radix) { throw new RuntimeException("Stub!"); }

/**
 * Returns the match result of the last scanning operation performed
 * by this scanner. This method throws {@code IllegalStateException}
 * if no match has been performed, or if the last match was
 * not successful.
 *
 * <p>The various {@code next} methods of {@code Scanner}
 * make a match result available if they complete without throwing an
 * exception. For instance, after an invocation of the {@link #nextInt}
 * method that returned an int, this method returns a
 * {@code MatchResult} for the search of the
 * <a href="#Integer-regex"><i>Integer</i></a> regular expression
 * defined above. Similarly the {@link #findInLine findInLine()},
 * {@link #findWithinHorizon findWithinHorizon()}, and {@link #skip skip()}
 * methods will make a match available if they succeed.
 *
 * @return a match result for the last match operation
 * @throws java.lang.IllegalStateException  If no match result is available
 */

public java.util.regex.MatchResult match() { throw new RuntimeException("Stub!"); }

/**
 * <p>Returns the string representation of this {@code Scanner}. The
 * string representation of a {@code Scanner} contains information
 * that may be useful for debugging. The exact format is unspecified.
 *
 * @return  The string representation of this scanner
 */

public java.lang.String toString() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if this scanner has another token in its input.
 * This method may block while waiting for input to scan.
 * The scanner does not advance past any input.
 *
 * @return true if and only if this scanner has another token
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @see java.util.Iterator
 */

public boolean hasNext() { throw new RuntimeException("Stub!"); }

/**
 * Finds and returns the next complete token from this scanner.
 * A complete token is preceded and followed by input that matches
 * the delimiter pattern. This method may block while waiting for input
 * to scan, even if a previous invocation of {@link #hasNext} returned
 * {@code true}.
 *
 * @return the next token
 * @throws java.util.NoSuchElementException if no more tokens are available
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @see java.util.Iterator
 */

public java.lang.String next() { throw new RuntimeException("Stub!"); }

/**
 * The remove operation is not supported by this implementation of
 * {@code Iterator}.
 *
 * @throws java.lang.UnsupportedOperationException if this method is invoked.
 * @see java.util.Iterator
 */

public void remove() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token matches the pattern constructed from the
 * specified string. The scanner does not advance past any input.
 *
 * <p> An invocation of this method of the form {@code hasNext(pattern)}
 * behaves in exactly the same way as the invocation
 * {@code hasNext(Pattern.compile(pattern))}.
 *
 * @param pattern a string specifying the pattern to scan
 * @return true if and only if this scanner has another token matching
 *         the specified pattern
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNext(java.lang.String pattern) { throw new RuntimeException("Stub!"); }

/**
 * Returns the next token if it matches the pattern constructed from the
 * specified string.  If the match is successful, the scanner advances
 * past the input that matched the pattern.
 *
 * <p> An invocation of this method of the form {@code next(pattern)}
 * behaves in exactly the same way as the invocation
 * {@code next(Pattern.compile(pattern))}.
 *
 * @param pattern a string specifying the pattern to scan
 * @return the next token
 * @throws java.util.NoSuchElementException if no such tokens are available
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public java.lang.String next(java.lang.String pattern) { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next complete token matches the specified pattern.
 * A complete token is prefixed and postfixed by input that matches
 * the delimiter pattern. This method may block while waiting for input.
 * The scanner does not advance past any input.
 *
 * @param pattern the pattern to scan for
 * @return true if and only if this scanner has another token matching
 *         the specified pattern
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNext(java.util.regex.Pattern pattern) { throw new RuntimeException("Stub!"); }

/**
 * Returns the next token if it matches the specified pattern. This
 * method may block while waiting for input to scan, even if a previous
 * invocation of {@link #hasNext(java.util.regex.Pattern)} returned {@code true}.
 * If the match is successful, the scanner advances past the input that
 * matched the pattern.
 *
 * @param pattern the pattern to scan for
 * @return the next token
 * @throws java.util.NoSuchElementException if no more tokens are available
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public java.lang.String next(java.util.regex.Pattern pattern) { throw new RuntimeException("Stub!"); }

/**
 * Returns true if there is another line in the input of this scanner.
 * This method may block while waiting for input. The scanner does not
 * advance past any input.
 *
 * @return true if and only if this scanner has another line of input
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNextLine() { throw new RuntimeException("Stub!"); }

/**
 * Advances this scanner past the current line and returns the input
 * that was skipped.
 *
 * This method returns the rest of the current line, excluding any line
 * separator at the end. The position is set to the beginning of the next
 * line.
 *
 * <p>Since this method continues to search through the input looking
 * for a line separator, it may buffer all of the input searching for
 * the line to skip if no line separators are present.
 *
 * @return the line that was skipped
 * @throws java.util.NoSuchElementException if no line was found
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public java.lang.String nextLine() { throw new RuntimeException("Stub!"); }

/**
 * Attempts to find the next occurrence of a pattern constructed from the
 * specified string, ignoring delimiters.
 *
 * <p>An invocation of this method of the form {@code findInLine(pattern)}
 * behaves in exactly the same way as the invocation
 * {@code findInLine(Pattern.compile(pattern))}.
 *
 * @param pattern a string specifying the pattern to search for
 * @return the text that matched the specified pattern
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public java.lang.String findInLine(java.lang.String pattern) { throw new RuntimeException("Stub!"); }

/**
 * Attempts to find the next occurrence of the specified pattern ignoring
 * delimiters. If the pattern is found before the next line separator, the
 * scanner advances past the input that matched and returns the string that
 * matched the pattern.
 * If no such pattern is detected in the input up to the next line
 * separator, then {@code null} is returned and the scanner's
 * position is unchanged. This method may block waiting for input that
 * matches the pattern.
 *
 * <p>Since this method continues to search through the input looking
 * for the specified pattern, it may buffer all of the input searching for
 * the desired token if no line separators are present.
 *
 * @param pattern the pattern to scan for
 * @return the text that matched the specified pattern
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public java.lang.String findInLine(java.util.regex.Pattern pattern) { throw new RuntimeException("Stub!"); }

/**
 * Attempts to find the next occurrence of a pattern constructed from the
 * specified string, ignoring delimiters.
 *
 * <p>An invocation of this method of the form
 * {@code findWithinHorizon(pattern)} behaves in exactly the same way as
 * the invocation
 * {@code findWithinHorizon(Pattern.compile(pattern), horizon)}.
 *
 * @param pattern a string specifying the pattern to search for
 * @param horizon the search horizon
 * @return the text that matched the specified pattern
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if horizon is negative
 */

public java.lang.String findWithinHorizon(java.lang.String pattern, int horizon) { throw new RuntimeException("Stub!"); }

/**
 * Attempts to find the next occurrence of the specified pattern.
 *
 * <p>This method searches through the input up to the specified
 * search horizon, ignoring delimiters. If the pattern is found the
 * scanner advances past the input that matched and returns the string
 * that matched the pattern. If no such pattern is detected then the
 * null is returned and the scanner's position remains unchanged. This
 * method may block waiting for input that matches the pattern.
 *
 * <p>A scanner will never search more than {@code horizon} code
 * points beyond its current position. Note that a match may be clipped
 * by the horizon; that is, an arbitrary match result may have been
 * different if the horizon had been larger. The scanner treats the
 * horizon as a transparent, non-anchoring bound (see {@link java.util.regex.Matcher#useTransparentBounds  } and {@link java.util.regex.Matcher#useAnchoringBounds Matcher#useAnchoringBounds}).
 *
 * <p>If horizon is {@code 0}, then the horizon is ignored and
 * this method continues to search through the input looking for the
 * specified pattern without bound. In this case it may buffer all of
 * the input searching for the pattern.
 *
 * <p>If horizon is negative, then an IllegalArgumentException is
 * thrown.
 *
 * @param pattern the pattern to scan for
 * @param horizon the search horizon
 * @return the text that matched the specified pattern
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if horizon is negative
 */

public java.lang.String findWithinHorizon(java.util.regex.Pattern pattern, int horizon) { throw new RuntimeException("Stub!"); }

/**
 * Skips input that matches the specified pattern, ignoring delimiters.
 * This method will skip input if an anchored match of the specified
 * pattern succeeds.
 *
 * <p>If a match to the specified pattern is not found at the
 * current position, then no input is skipped and a
 * {@code NoSuchElementException} is thrown.
 *
 * <p>Since this method seeks to match the specified pattern starting at
 * the scanner's current position, patterns that can match a lot of
 * input (".*", for example) may cause the scanner to buffer a large
 * amount of input.
 *
 * <p>Note that it is possible to skip something without risking a
 * {@code NoSuchElementException} by using a pattern that can
 * match nothing, e.g., {@code sc.skip("[ \t]*")}.
 *
 * @param pattern a string specifying the pattern to skip over
 * @return this scanner
 * @throws java.util.NoSuchElementException if the specified pattern is not found
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public java.util.Scanner skip(java.util.regex.Pattern pattern) { throw new RuntimeException("Stub!"); }

/**
 * Skips input that matches a pattern constructed from the specified
 * string.
 *
 * <p> An invocation of this method of the form {@code skip(pattern)}
 * behaves in exactly the same way as the invocation
 * {@code skip(Pattern.compile(pattern))}.
 *
 * @param pattern a string specifying the pattern to skip over
 * @return this scanner
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public java.util.Scanner skip(java.lang.String pattern) { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a boolean value using a case insensitive pattern
 * created from the string "true|false".  The scanner does not
 * advance past the input that matched.
 *
 * @return true if and only if this scanner's next token is a valid
 *         boolean value
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNextBoolean() { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input into a boolean value and returns
 * that value. This method will throw {@code InputMismatchException}
 * if the next token cannot be translated into a valid boolean value.
 * If the match is successful, the scanner advances past the input that
 * matched.
 *
 * @return the boolean scanned from the input
 * @throws java.util.InputMismatchException if the next token is not a valid boolean
 * @throws java.util.NoSuchElementException if input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean nextBoolean() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a byte value in the default radix using the
 * {@link #nextByte} method. The scanner does not advance past any input.
 *
 * @return true if and only if this scanner's next token is a valid
 *         byte value
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNextByte() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a byte value in the specified radix using the
 * {@link #nextByte} method. The scanner does not advance past any input.
 *
 * <p>If the radix is less than {@link java.lang.Character#MIN_RADIX Character.MIN_RADIX}
 * or greater than {@link java.lang.Character#MAX_RADIX Character.MAX_RADIX}, then an
 * {@code IllegalArgumentException} is thrown.
 *
 * @param radix the radix used to interpret the token as a byte value
 * @return true if and only if this scanner's next token is a valid
 *         byte value
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if the radix is out of range
 */

public boolean hasNextByte(int radix) { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as a {@code byte}.
 *
 * <p> An invocation of this method of the form
 * {@code nextByte()} behaves in exactly the same way as the
 * invocation {@code nextByte(radix)}, where {@code radix}
 * is the default radix of this scanner.
 *
 * @return the {@code byte} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Integer</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public byte nextByte() { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as a {@code byte}.
 * This method will throw {@code InputMismatchException}
 * if the next token cannot be translated into a valid byte value as
 * described below. If the translation is successful, the scanner advances
 * past the input that matched.
 *
 * <p> If the next token matches the <a
 * href="#Integer-regex"><i>Integer</i></a> regular expression defined
 * above then the token is converted into a {@code byte} value as if by
 * removing all locale specific prefixes, group separators, and locale
 * specific suffixes, then mapping non-ASCII digits into ASCII
 * digits via {@link java.lang.Character#digit Character.digit}, prepending a
 * negative sign (-) if the locale specific negative prefixes and suffixes
 * were present, and passing the resulting string to
 * {@link java.lang.Byte#parseByte(java.lang.String,int) Byte.parseByte} with the
 * specified radix.
 *
 * <p>If the radix is less than {@link java.lang.Character#MIN_RADIX Character.MIN_RADIX}
 * or greater than {@link java.lang.Character#MAX_RADIX Character.MAX_RADIX}, then an
 * {@code IllegalArgumentException} is thrown.
 *
 * @param radix the radix used to interpret the token as a byte value
 * @return the {@code byte} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Integer</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if the radix is out of range
 */

public byte nextByte(int radix) { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a short value in the default radix using the
 * {@link #nextShort} method. The scanner does not advance past any input.
 *
 * @return true if and only if this scanner's next token is a valid
 *         short value in the default radix
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNextShort() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a short value in the specified radix using the
 * {@link #nextShort} method. The scanner does not advance past any input.
 *
 * <p>If the radix is less than {@link java.lang.Character#MIN_RADIX Character.MIN_RADIX}
 * or greater than {@link java.lang.Character#MAX_RADIX Character.MAX_RADIX}, then an
 * {@code IllegalArgumentException} is thrown.
 *
 * @param radix the radix used to interpret the token as a short value
 * @return true if and only if this scanner's next token is a valid
 *         short value in the specified radix
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if the radix is out of range
 */

public boolean hasNextShort(int radix) { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as a {@code short}.
 *
 * <p> An invocation of this method of the form
 * {@code nextShort()} behaves in exactly the same way as the
 * invocation {@link #nextShort(int) nextShort(radix)}, where {@code radix}
 * is the default radix of this scanner.
 *
 * @return the {@code short} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Integer</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public short nextShort() { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as a {@code short}.
 * This method will throw {@code InputMismatchException}
 * if the next token cannot be translated into a valid short value as
 * described below. If the translation is successful, the scanner advances
 * past the input that matched.
 *
 * <p> If the next token matches the <a
 * href="#Integer-regex"><i>Integer</i></a> regular expression defined
 * above then the token is converted into a {@code short} value as if by
 * removing all locale specific prefixes, group separators, and locale
 * specific suffixes, then mapping non-ASCII digits into ASCII
 * digits via {@link java.lang.Character#digit Character.digit}, prepending a
 * negative sign (-) if the locale specific negative prefixes and suffixes
 * were present, and passing the resulting string to
 * {@link java.lang.Short#parseShort(java.lang.String,int) Short.parseShort} with the
 * specified radix.
 *
 * <p>If the radix is less than {@link java.lang.Character#MIN_RADIX Character.MIN_RADIX}
 * or greater than {@link java.lang.Character#MAX_RADIX Character.MAX_RADIX}, then an
 * {@code IllegalArgumentException} is thrown.
 *
 * @param radix the radix used to interpret the token as a short value
 * @return the {@code short} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Integer</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if the radix is out of range
 */

public short nextShort(int radix) { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as an int value in the default radix using the
 * {@link #nextInt} method. The scanner does not advance past any input.
 *
 * @return true if and only if this scanner's next token is a valid
 *         int value
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNextInt() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as an int value in the specified radix using the
 * {@link #nextInt} method. The scanner does not advance past any input.
 *
 * <p>If the radix is less than {@link java.lang.Character#MIN_RADIX Character.MIN_RADIX}
 * or greater than {@link java.lang.Character#MAX_RADIX Character.MAX_RADIX}, then an
 * {@code IllegalArgumentException} is thrown.
 *
 * @param radix the radix used to interpret the token as an int value
 * @return true if and only if this scanner's next token is a valid
 *         int value
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if the radix is out of range
 */

public boolean hasNextInt(int radix) { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as an {@code int}.
 *
 * <p> An invocation of this method of the form
 * {@code nextInt()} behaves in exactly the same way as the
 * invocation {@code nextInt(radix)}, where {@code radix}
 * is the default radix of this scanner.
 *
 * @return the {@code int} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Integer</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public int nextInt() { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as an {@code int}.
 * This method will throw {@code InputMismatchException}
 * if the next token cannot be translated into a valid int value as
 * described below. If the translation is successful, the scanner advances
 * past the input that matched.
 *
 * <p> If the next token matches the <a
 * href="#Integer-regex"><i>Integer</i></a> regular expression defined
 * above then the token is converted into an {@code int} value as if by
 * removing all locale specific prefixes, group separators, and locale
 * specific suffixes, then mapping non-ASCII digits into ASCII
 * digits via {@link java.lang.Character#digit Character.digit}, prepending a
 * negative sign (-) if the locale specific negative prefixes and suffixes
 * were present, and passing the resulting string to
 * {@link java.lang.Integer#parseInt(java.lang.String,int) Integer.parseInt} with the
 * specified radix.
 *
 * <p>If the radix is less than {@link java.lang.Character#MIN_RADIX Character.MIN_RADIX}
 * or greater than {@link java.lang.Character#MAX_RADIX Character.MAX_RADIX}, then an
 * {@code IllegalArgumentException} is thrown.
 *
 * @param radix the radix used to interpret the token as an int value
 * @return the {@code int} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Integer</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if the radix is out of range
 */

public int nextInt(int radix) { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a long value in the default radix using the
 * {@link #nextLong} method. The scanner does not advance past any input.
 *
 * @return true if and only if this scanner's next token is a valid
 *         long value
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNextLong() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a long value in the specified radix using the
 * {@link #nextLong} method. The scanner does not advance past any input.
 *
 * <p>If the radix is less than {@link java.lang.Character#MIN_RADIX Character.MIN_RADIX}
 * or greater than {@link java.lang.Character#MAX_RADIX Character.MAX_RADIX}, then an
 * {@code IllegalArgumentException} is thrown.
 *
 * @param radix the radix used to interpret the token as a long value
 * @return true if and only if this scanner's next token is a valid
 *         long value
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if the radix is out of range
 */

public boolean hasNextLong(int radix) { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as a {@code long}.
 *
 * <p> An invocation of this method of the form
 * {@code nextLong()} behaves in exactly the same way as the
 * invocation {@code nextLong(radix)}, where {@code radix}
 * is the default radix of this scanner.
 *
 * @return the {@code long} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Integer</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public long nextLong() { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as a {@code long}.
 * This method will throw {@code InputMismatchException}
 * if the next token cannot be translated into a valid long value as
 * described below. If the translation is successful, the scanner advances
 * past the input that matched.
 *
 * <p> If the next token matches the <a
 * href="#Integer-regex"><i>Integer</i></a> regular expression defined
 * above then the token is converted into a {@code long} value as if by
 * removing all locale specific prefixes, group separators, and locale
 * specific suffixes, then mapping non-ASCII digits into ASCII
 * digits via {@link java.lang.Character#digit Character.digit}, prepending a
 * negative sign (-) if the locale specific negative prefixes and suffixes
 * were present, and passing the resulting string to
 * {@link java.lang.Long#parseLong(java.lang.String,int) Long.parseLong} with the
 * specified radix.
 *
 * <p>If the radix is less than {@link java.lang.Character#MIN_RADIX Character.MIN_RADIX}
 * or greater than {@link java.lang.Character#MAX_RADIX Character.MAX_RADIX}, then an
 * {@code IllegalArgumentException} is thrown.
 *
 * @param radix the radix used to interpret the token as an int value
 * @return the {@code long} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Integer</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if the radix is out of range
 */

public long nextLong(int radix) { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a float value using the {@link #nextFloat}
 * method. The scanner does not advance past any input.
 *
 * @return true if and only if this scanner's next token is a valid
 *         float value
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNextFloat() { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as a {@code float}.
 * This method will throw {@code InputMismatchException}
 * if the next token cannot be translated into a valid float value as
 * described below. If the translation is successful, the scanner advances
 * past the input that matched.
 *
 * <p> If the next token matches the <a
 * href="#Float-regex"><i>Float</i></a> regular expression defined above
 * then the token is converted into a {@code float} value as if by
 * removing all locale specific prefixes, group separators, and locale
 * specific suffixes, then mapping non-ASCII digits into ASCII
 * digits via {@link java.lang.Character#digit Character.digit}, prepending a
 * negative sign (-) if the locale specific negative prefixes and suffixes
 * were present, and passing the resulting string to
 * {@link java.lang.Float#parseFloat Float.parseFloat}. If the token matches
 * the localized NaN or infinity strings, then either "Nan" or "Infinity"
 * is passed to {@link java.lang.Float#parseFloat(java.lang.String) Float.parseFloat} as
 * appropriate.
 *
 * @return the {@code float} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Float</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public float nextFloat() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a double value using the {@link #nextDouble}
 * method. The scanner does not advance past any input.
 *
 * @return true if and only if this scanner's next token is a valid
 *         double value
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNextDouble() { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as a {@code double}.
 * This method will throw {@code InputMismatchException}
 * if the next token cannot be translated into a valid double value.
 * If the translation is successful, the scanner advances past the input
 * that matched.
 *
 * <p> If the next token matches the <a
 * href="#Float-regex"><i>Float</i></a> regular expression defined above
 * then the token is converted into a {@code double} value as if by
 * removing all locale specific prefixes, group separators, and locale
 * specific suffixes, then mapping non-ASCII digits into ASCII
 * digits via {@link java.lang.Character#digit Character.digit}, prepending a
 * negative sign (-) if the locale specific negative prefixes and suffixes
 * were present, and passing the resulting string to
 * {@link java.lang.Double#parseDouble Double.parseDouble}. If the token matches
 * the localized NaN or infinity strings, then either "Nan" or "Infinity"
 * is passed to {@link java.lang.Double#parseDouble(java.lang.String) Double.parseDouble} as
 * appropriate.
 *
 * @return the {@code double} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Float</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if the input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public double nextDouble() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a {@code BigInteger} in the default radix using the
 * {@link #nextBigInteger} method. The scanner does not advance past any
 * input.
 *
 * @return true if and only if this scanner's next token is a valid
 *         {@code BigInteger}
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNextBigInteger() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a {@code BigInteger} in the specified radix using
 * the {@link #nextBigInteger} method. The scanner does not advance past
 * any input.
 *
 * <p>If the radix is less than {@link java.lang.Character#MIN_RADIX Character.MIN_RADIX}
 * or greater than {@link java.lang.Character#MAX_RADIX Character.MAX_RADIX}, then an
 * {@code IllegalArgumentException} is thrown.
 *
 * @param radix the radix used to interpret the token as an integer
 * @return true if and only if this scanner's next token is a valid
 *         {@code BigInteger}
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if the radix is out of range
 */

public boolean hasNextBigInteger(int radix) { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as a {@link java.math.BigInteger
 * BigInteger}.
 *
 * <p> An invocation of this method of the form
 * {@code nextBigInteger()} behaves in exactly the same way as the
 * invocation {@code nextBigInteger(radix)}, where {@code radix}
 * is the default radix of this scanner.
 *
 * @return the {@code BigInteger} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Integer</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if the input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public java.math.BigInteger nextBigInteger() { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as a {@link java.math.BigInteger
 * BigInteger}.
 *
 * <p> If the next token matches the <a
 * href="#Integer-regex"><i>Integer</i></a> regular expression defined
 * above then the token is converted into a {@code BigInteger} value as if
 * by removing all group separators, mapping non-ASCII digits into ASCII
 * digits via the {@link java.lang.Character#digit Character.digit}, and passing the
 * resulting string to the {@link
 * java.math.BigInteger#BigInteger(java.lang.String)
 * BigInteger(String, int)} constructor with the specified radix.
 *
 * <p>If the radix is less than {@link java.lang.Character#MIN_RADIX Character.MIN_RADIX}
 * or greater than {@link java.lang.Character#MAX_RADIX Character.MAX_RADIX}, then an
 * {@code IllegalArgumentException} is thrown.
 *
 * @param radix the radix used to interpret the token
 * @return the {@code BigInteger} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Integer</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if the input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.lang.IllegalArgumentException if the radix is out of range
 */

public java.math.BigInteger nextBigInteger(int radix) { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the next token in this scanner's input can be
 * interpreted as a {@code BigDecimal} using the
 * {@link #nextBigDecimal} method. The scanner does not advance past any
 * input.
 *
 * @return true if and only if this scanner's next token is a valid
 *         {@code BigDecimal}
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public boolean hasNextBigDecimal() { throw new RuntimeException("Stub!"); }

/**
 * Scans the next token of the input as a {@link java.math.BigDecimal
 * BigDecimal}.
 *
 * <p> If the next token matches the <a
 * href="#Decimal-regex"><i>Decimal</i></a> regular expression defined
 * above then the token is converted into a {@code BigDecimal} value as if
 * by removing all group separators, mapping non-ASCII digits into ASCII
 * digits via the {@link java.lang.Character#digit Character.digit}, and passing the
 * resulting string to the {@link
 * java.math.BigDecimal#BigDecimal(java.lang.String) BigDecimal(String)}
 * constructor.
 *
 * @return the {@code BigDecimal} scanned from the input
 * @throws java.util.InputMismatchException
 *         if the next token does not match the <i>Decimal</i>
 *         regular expression, or is out of range
 * @throws java.util.NoSuchElementException if the input is exhausted
 * @throws java.lang.IllegalStateException if this scanner is closed
 */

public java.math.BigDecimal nextBigDecimal() { throw new RuntimeException("Stub!"); }

/**
 * Resets this scanner.
 *
 * <p> Resetting a scanner discards all of its explicit state
 * information which may have been changed by invocations of
 * {@link #useDelimiter useDelimiter()},
 * {@link #useLocale useLocale()}, or
 * {@link #useRadix useRadix()}.
 *
 * <p> An invocation of this method of the form
 * {@code scanner.reset()} behaves in exactly the same way as the
 * invocation
 *
 * <blockquote><pre>{@code
 *   scanner.useDelimiter("\\p{javaWhitespace}+")
 *          .useLocale(Locale.getDefault(Locale.Category.FORMAT))
 *          .useRadix(10);
 * }</pre></blockquote>
 *
 * @return this scanner
 *
 * @since 1.6
 */

public java.util.Scanner reset() { throw new RuntimeException("Stub!"); }

/**
 * Returns a stream of delimiter-separated tokens from this scanner. The
 * stream contains the same tokens that would be returned, starting from
 * this scanner's current state, by calling the {@link #next} method
 * repeatedly until the {@link #hasNext} method returns false.
 *
 * <p>The resulting stream is sequential and ordered. All stream elements are
 * non-null.
 *
 * <p>Scanning starts upon initiation of the terminal stream operation, using the
 * current state of this scanner. Subsequent calls to any methods on this scanner
 * other than {@link #close} and {@link #ioException} may return undefined results
 * or may cause undefined effects on the returned stream. The returned stream's source
 * {@code Spliterator} is <em>fail-fast</em> and will, on a best-effort basis, throw a
 * {@link java.util.ConcurrentModificationException} if any such calls are detected
 * during stream pipeline execution.
 *
 * <p>After stream pipeline execution completes, this scanner is left in an indeterminate
 * state and cannot be reused.
 *
 * <p>If this scanner contains a resource that must be released, this scanner
 * should be closed, either by calling its {@link #close} method, or by
 * closing the returned stream. Closing the stream will close the underlying scanner.
 * {@code IllegalStateException} is thrown if the scanner has been closed when this
 * method is called, or if this scanner is closed during stream pipeline execution.
 *
 * <p>This method might block waiting for more input.
 *
 * @apiNote
 * For example, the following code will create a list of
 * comma-delimited tokens from a string:
 *
 * <pre>{@code
 * List<String> result = new Scanner("abc,def,,ghi")
 *     .useDelimiter(",")
 *     .tokens()
 *     .collect(Collectors.toList());
 * }</pre>
 *
 * <p>The resulting list would contain {@code "abc"}, {@code "def"},
 * the empty string, and {@code "ghi"}.
 *
 * @return a sequential stream of token strings
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @since 9
 */

public java.util.stream.Stream<java.lang.String> tokens() { throw new RuntimeException("Stub!"); }

/**
 * Returns a stream of match results from this scanner. The stream
 * contains the same results in the same order that would be returned by
 * calling {@code findWithinHorizon(pattern, 0)} and then {@link #match}
 * successively as long as {@link #findWithinHorizon findWithinHorizon()}
 * finds matches.
 *
 * <p>The resulting stream is sequential and ordered. All stream elements are
 * non-null.
 *
 * <p>Scanning starts upon initiation of the terminal stream operation, using the
 * current state of this scanner. Subsequent calls to any methods on this scanner
 * other than {@link #close} and {@link #ioException} may return undefined results
 * or may cause undefined effects on the returned stream. The returned stream's source
 * {@code Spliterator} is <em>fail-fast</em> and will, on a best-effort basis, throw a
 * {@link java.util.ConcurrentModificationException} if any such calls are detected
 * during stream pipeline execution.
 *
 * <p>After stream pipeline execution completes, this scanner is left in an indeterminate
 * state and cannot be reused.
 *
 * <p>If this scanner contains a resource that must be released, this scanner
 * should be closed, either by calling its {@link #close} method, or by
 * closing the returned stream. Closing the stream will close the underlying scanner.
 * {@code IllegalStateException} is thrown if the scanner has been closed when this
 * method is called, or if this scanner is closed during stream pipeline execution.
 *
 * <p>As with the {@link #findWithinHorizon findWithinHorizon()} methods, this method
 * might block waiting for additional input, and it might buffer an unbounded amount of
 * input searching for a match.
 *
 * @apiNote
 * For example, the following code will read a file and return a list
 * of all sequences of characters consisting of seven or more Latin capital
 * letters:
 *
 * <pre>{@code
 * try (Scanner sc = new Scanner(Path.of("input.txt"))) {
 *     Pattern pat = Pattern.compile("[A-Z]{7,}");
 *     List<String> capWords = sc.findAll(pat)
 *                               .map(MatchResult::group)
 *                               .collect(Collectors.toList());
 * }
 * }</pre>
 *
 * @param pattern the pattern to be matched
 * @return a sequential stream of match results
 * @throws java.lang.NullPointerException if pattern is null
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @since 9
 */

public java.util.stream.Stream<java.util.regex.MatchResult> findAll(java.util.regex.Pattern pattern) { throw new RuntimeException("Stub!"); }

/**
 * Returns a stream of match results that match the provided pattern string.
 * The effect is equivalent to the following code:
 *
 * <pre>{@code
 *     scanner.findAll(Pattern.compile(patString))
 * }</pre>
 *
 * @param patString the pattern string
 * @return a sequential stream of match results
 * @throws java.lang.NullPointerException if patString is null
 * @throws java.lang.IllegalStateException if this scanner is closed
 * @throws java.util.regex.PatternSyntaxException if the regular expression's syntax is invalid
 * @since 9
 * @see java.util.regex.Pattern
 */

public java.util.stream.Stream<java.util.regex.MatchResult> findAll(java.lang.String patString) { throw new RuntimeException("Stub!"); }
}

